// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.contexts.model;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.NotNullLazyValue;
import com.intellij.psi.PsiElement;
import com.intellij.psi.util.CachedValue;
import com.intellij.psi.util.CachedValueProvider.Result;
import com.intellij.psi.util.CachedValuesManager;
import com.intellij.spring.model.BeanService;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.utils.SpringProfileUtils;
import com.intellij.spring.model.xml.context.SpringBeansPackagesScan;
import org.jetbrains.annotations.NotNull;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

public class SpringComponentScanModel<T extends SpringBeansPackagesScan> extends ComponentScanPackagesModel {

  private final Set<String> myActiveProfiles;
  private volatile CachedValue<Collection<SpringBeanPointer<?>>> myScannedBeans;

  private final T myComponentScan;

  public SpringComponentScanModel(@NotNull Module module, @NotNull final T componentScan, Set<String> activeProfiles) {
    super(NotNullLazyValue.lazy(() -> {
      return componentScan.getPsiPackages();
    }), module);
    myComponentScan = componentScan;
    myActiveProfiles = activeProfiles;
  }

  @Override
  public Collection<SpringBeanPointer<?>> getLocalBeans() {
    PsiElement psiElement = myComponentScan.getIdentifyingPsiElement();
    if (psiElement == null) return Collections.emptySet();

    if (myScannedBeans == null) {
      myScannedBeans = CachedValuesManager.getManager(getModule().getProject()).createCachedValue(() -> {
        Collection<SpringBeanPointer<?>> pointers = calculateLocalBeans();

        return Result.create(pointers,
                             getDependencies(pointers.stream().map(pointer -> pointer.getContainingFile()).collect(Collectors.toSet())));
      }, false);
    }

    return myScannedBeans.getValue();
  }

  @Override
  protected Collection<SpringBeanPointer<?>> calculateScannedBeans() {
    final Set<CommonSpringBean> elements = myComponentScan.getScannedElements((getModule()));

    List<CommonSpringBean> inActiveProfiles = SpringProfileUtils.filterBeansInActiveProfiles(elements, myActiveProfiles);

    return BeanService.getInstance().mapSpringBeans(inActiveProfiles);
  }

  @NotNull
  @Override
  public Set<String> getActiveProfiles() {
    return myActiveProfiles;
  }
}
