// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.contexts.model;

import com.intellij.openapi.util.Condition;
import com.intellij.openapi.util.Conditions;
import com.intellij.openapi.util.Ref;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.SpringModelSearchParameters;
import com.intellij.spring.model.utils.SpringProfileUtils;
import com.intellij.util.CommonProcessors;
import com.intellij.util.Processor;
import com.intellij.util.SystemProperties;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Set;

public abstract class SpringCachingProcessor<InParams extends SpringModelSearchParameters> {
  private static final int DEFAULT_CACHE_SIZE = 42;
  private static final String CACHE_SIZE_PROPERTY_NAME = "idea.spring.model.processing.cache.size";
  private static final int CONFIGURED_CACHE_SIZE = SystemProperties.getIntProperty(CACHE_SIZE_PROPERTY_NAME, DEFAULT_CACHE_SIZE);

  private final SpringSizeLimitedCache<InParams, Collection<SpringBeanPointer<?>>> myFindAllCache;
  private final SpringSizeLimitedCache<InParams, Ref<SpringBeanPointer<?>>> myFindFirstCache;

  protected SpringCachingProcessor() {
    this(Conditions.alwaysTrue());
  }

  protected SpringCachingProcessor(@NotNull Condition<? super InParams> keyValidityCheck) {
    myFindAllCache = new SpringSizeLimitedCache<>(CONFIGURED_CACHE_SIZE, keyValidityCheck) {
      @NotNull
      @Override
      protected Collection<SpringBeanPointer<?>> createValue(InParams key) {
        return findPointers(key);
      }
    };

    myFindFirstCache = new SpringSizeLimitedCache<>(CONFIGURED_CACHE_SIZE, keyValidityCheck) {
      @NotNull
      @Override
      protected Ref<SpringBeanPointer<?>> createValue(InParams key) {
        return Ref.create(findFirstPointer(key));
      }
    };
  }

  @NotNull
  protected abstract Collection<SpringBeanPointer<?>> findPointers(@NotNull InParams parameters);

  @Nullable
  protected abstract SpringBeanPointer<?> findFirstPointer(@NotNull InParams parameters);

  public boolean process(InParams params,
                         final Processor<? super SpringBeanPointer<?>> processor,
                         final Set<String> activeProfiles) {
    if (processor instanceof CommonProcessors.FindFirstProcessor && myFindAllCache.getCachedValue(params) == null) {
      final SpringBeanPointer<?> first = myFindFirstCache.get(params).get();
      return processBeansInActiveProfile(processor, first, activeProfiles);
    }

    for (SpringBeanPointer pointer : myFindAllCache.get(params)) {
      if (!processBeansInActiveProfile(processor, pointer, activeProfiles)) return false;
    }
    return true;
  }

  private static boolean processBeansInActiveProfile(Processor<? super SpringBeanPointer<?>> processor,
                                                     @Nullable SpringBeanPointer<?> pointer,
                                                     Set<String> activeProfiles) {
    if (pointer == null) return true;
    if (ContainerUtil.isEmpty(activeProfiles)) return processor.process(pointer);

    if (!SpringProfileUtils.isInActiveProfiles(pointer.getSpringBean(), activeProfiles)) return true;

    return processor.process(pointer);
  }
}
