// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.contexts.model;

import com.intellij.jam.JamService;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.Pair;
import com.intellij.psi.*;
import com.intellij.psi.impl.compiled.ClsFileImpl;
import com.intellij.psi.util.CachedValue;
import com.intellij.psi.util.CachedValueProvider;
import com.intellij.psi.util.CachedValuesManager;
import com.intellij.psi.util.InheritanceUtil;
import com.intellij.psi.xml.XmlFile;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.SpringLocalModelFactory;
import com.intellij.spring.SpringModificationTrackersManager;
import com.intellij.spring.contexts.model.graph.LocalModelDependency;
import com.intellij.spring.contexts.model.graph.LocalModelDependencyType;
import com.intellij.spring.model.*;
import com.intellij.spring.model.jam.JamPsiMemberSpringBean;
import com.intellij.spring.model.jam.javaConfig.ContextJavaBean;
import com.intellij.spring.model.jam.javaConfig.SpringJavaBean;
import com.intellij.spring.model.jam.stereotype.CustomSpringComponent;
import com.intellij.spring.model.jam.stereotype.SpringConfiguration;
import com.intellij.spring.model.jam.stereotype.SpringStereotypeElement;
import com.intellij.spring.model.jam.utils.SpringJamUtils;
import com.intellij.spring.model.utils.SpringCommonUtils;
import com.intellij.spring.model.utils.SpringProfileUtils;
import com.intellij.spring.model.xml.context.SpringBeansPackagesScan;
import com.intellij.util.ArrayUtil;
import com.intellij.util.SmartList;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.intellij.jam.JamService.CHECK_DEEP;
import static com.intellij.jam.JamService.CHECK_METHOD;

public abstract class LocalAnnotationModel extends AbstractSimpleLocalModel<PsiClass> {
  @NotNull private final PsiClass myClass;
  @NotNull protected final Module myModule;
  @NotNull protected final Set<String> myActiveProfiles;
  @NotNull private final CachedValue<Collection<ContextJavaBean>> myLocalContextBeansCachedValue;

  public LocalAnnotationModel(@NotNull PsiClass aClass, @NotNull Module module, @NotNull Set<String> activeProfiles) {
    myClass = aClass;
    myModule = module;
    myActiveProfiles = Set.copyOf(activeProfiles);
    myLocalContextBeansCachedValue = CachedValuesManager.getManager(myClass.getProject()).createCachedValue(() -> {
      List<ContextJavaBean> beans = JamService.getJamService(myClass.getProject())
        .getAnnotatedMembersList(myClass, ContextJavaBean.BEAN_JAM_KEY, CHECK_METHOD | CHECK_DEEP);
      Set<PsiFile> dependencies = ContainerUtil.newHashSet(myClass.getContainingFile());
      for (ContextJavaBean bean : beans) {
        if (bean.isValid()) {
          ContainerUtil.addIfNotNull(dependencies, bean.getContainingFile());
        }
      }
      return CachedValueProvider.Result.create(beans, dependencies);
    });
  }

  @Override
  @NotNull
  public PsiClass getConfig() {
    return myClass;
  }

  @Override
  public Collection<SpringBeanPointer<?>> getLocalBeans() {
    Collection<CommonSpringBean> allBeans = new SmartList<>();

    ContainerUtil.addIfNotNull(allBeans, getBeanForClass(myClass));
    ContainerUtil.addAllNotNull(allBeans, SpringProfileUtils
      .filterBeansInActiveProfiles(myLocalContextBeansCachedValue.getValue(), getActiveProfiles()));

    return BeanService.getInstance().mapSpringBeans(allBeans);
  }

  @Nullable
  private static CommonSpringBean getBeanForClass(@NotNull PsiClass aClass) {
    return CachedValuesManager.getCachedValue(aClass, () -> {
      CommonSpringBean commonSpringBean =
        JamService.getJamService(aClass.getProject()).getJamElement(JamPsiMemberSpringBean.PSI_MEMBER_SPRING_BEAN_JAM_KEY, aClass);
      if (commonSpringBean == null) {
        if (!aClass.isInterface() && !aClass.hasModifierProperty(PsiModifier.ABSTRACT) &&
            SpringCommonUtils.isSpringBeanCandidateClass(aClass)) {
          commonSpringBean = new CustomSpringComponent(aClass);
        }
      }
      return CachedValueProvider.Result.createSingleDependency(commonSpringBean, aClass);
    });
  }

  @Override
  public Set<CommonSpringModel> getRelatedModels() {
    Set<CommonSpringModel> models = new LinkedHashSet<>();

    ContainerUtil.addAllNotNull(models, getRelatedLocalModels());
    ContainerUtil.addAllNotNull(models, getCachedPackageScanModel());
    ContainerUtil.addAllNotNull(models, getCachedInnerStaticClassConfigurations());
    ContainerUtil.addAllNotNull(models, getCustomDiscoveredBeansModel());

    return models;
  }

  @NotNull
  private Set<CommonSpringModel> getCachedPackageScanModel() {
    return CachedValuesManager.getManager(getConfig().getProject()).getCachedValue(this, () -> CachedValueProvider.Result
      .create(getPackageScanModels(this), getOutsideModelDependencies(this)));
  }


  @NotNull
  protected Set<CommonSpringModel> getPackageScanModels(@NotNull LocalAnnotationModel localModel) {
    Set<CommonSpringModel> models = new LinkedHashSet<>();
    Module module = localModel.getModule();
    for (SpringBeansPackagesScan scan : localModel.getPackagesScans()) {
      models.add(new SpringComponentScanModel<>(module, scan, localModel.getActiveProfiles()));
    }

    return models;
  }

  @NotNull
  private Set<CommonSpringModel> getCachedInnerStaticClassConfigurations() {
    return CachedValuesManager.getManager(getConfig().getProject()).getCachedValue(this, () -> CachedValueProvider.Result
      .create(getInnerStaticClassConfigurations(getConfig()), getOutsideModelDependencies(this)));
  }

  private Set<CommonSpringModel> getInnerStaticClassConfigurations(@NotNull PsiClass config) {
    return getInnerStaticClassModels(config, aClass -> getLocalAnnotationModel(aClass));
  }

  public static Set<CommonSpringModel> getInnerStaticClassModels(@NotNull PsiClass config,
                                                                 Function<? super PsiClass, ? extends CommonSpringModel> mapper) {
    return Arrays.stream(config.getAllInnerClasses())
      .filter(psiClass -> psiClass.hasModifierProperty(PsiModifier.STATIC)
                          && JamService.getJamService(psiClass.getProject()).getJamElement(SpringConfiguration.JAM_KEY, psiClass) != null)
      .map(mapper)
      .collect(Collectors.toSet());
  }

  @Override
  @NotNull
  public Module getModule() {
    return myModule;
  }

  @NotNull
  @Override
  public Set<String> getProfiles() {
    Set<String> allProfiles = new LinkedHashSet<>();
    SpringConfiguration configuration = getConfiguration();
    if (configuration != null) {
      allProfiles.addAll(configuration.getProfile().getNames());
      for (SpringJavaBean bean : configuration.getBeans()) {
        allProfiles.addAll(bean.getProfile().getNames());
      }
    }
    return allProfiles;
  }

  @Nullable
  private SpringConfiguration getConfiguration() {
    return JamService.getJamService(myClass.getProject()).getJamElement(SpringConfiguration.JAM_KEY, myClass);
  }

  @NotNull
  @Override
  public List<SpringBeansPackagesScan> getPackagesScans() {
    List<SpringBeansPackagesScan> packageScans = new SmartList<>();

    packageScans.addAll(SpringJamUtils.getInstance().getBeansPackagesScan(myClass));
    for (PsiClass superClass : InheritanceUtil.getSuperClasses(myClass)) {
      if (CommonClassNames.JAVA_LANG_OBJECT.equals(superClass.getQualifiedName())) continue;
      packageScans.addAll(SpringJamUtils.getInstance().getBeansPackagesScan(superClass));
    }
    return packageScans;
  }

  @NotNull
  @Override
  public Set<String> getActiveProfiles() {
    return myActiveProfiles;
  }

  @NotNull
  @Override
  public Set<Pair<LocalModel, LocalModelDependency>> getDependentLocalModels() {
    return
      CachedValuesManager.getManager(getConfig().getProject()).getCachedValue(this, () -> {
        Module module = getModule();
        Set<Pair<LocalModel, LocalModelDependency>> models = new HashSet<>();
        if (!module.isDisposed()) {
          collectImportDependentLocalModels(models);

          collectScanDependentLocalModels(models);

          SpringJamUtils.getInstance().processCustomAnnotations(myClass, enableAnnotation -> {
            addNotNullModel(models, getLocalAnnotationModel(enableAnnotation.getFirst()), enableAnnotation.getSecond());
            return true;
          });

          SpringJamUtils.getInstance().processCustomDependentLocalModels(this, (model, dependency) -> {
            addNotNullModel(models, model, dependency);
            return true;
          });
        }
        final Set<Object> dependencies = new LinkedHashSet<>();
        ContainerUtil.addAll(dependencies, getOutsideModelDependencies(this));
        dependencies.addAll(models.stream().map(pair -> pair.first.getConfig()).collect(Collectors.toSet()));

        return CachedValueProvider.Result.create(models, ArrayUtil.toObjectArray(dependencies));
      });
  }

  private void collectImportDependentLocalModels(Set<Pair<LocalModel, LocalModelDependency>> models) {
    Module module = getModule();
    collectImportDependentLocalModels(models, module, myClass);
    for (PsiClass superClass : InheritanceUtil.getSuperClasses(myClass)) {
      if (CommonClassNames.JAVA_LANG_OBJECT.equals(superClass.getQualifiedName())) continue;

      collectImportDependentLocalModels(models, module, superClass);
    }
  }

  private void collectImportDependentLocalModels(Set<Pair<LocalModel, LocalModelDependency>> models, Module module, PsiClass psiClass) {
    SpringJamUtils.getInstance().processImportedResources(psiClass, pair -> {
      for (XmlFile xmlFile : pair.first) {
        addNotNullModel(models,
                        SpringLocalModelFactory.getInstance().getOrCreateLocalXmlModel(xmlFile, module, getActiveProfiles()),
                        LocalModelDependency.create(LocalModelDependencyType.IMPORT, pair.second));
      }
      return true;
    }, myModule);

    SpringJamUtils.getInstance().processImportedClasses(psiClass, pair -> {
      addNotNullModel(models,
                      getLocalAnnotationModel(pair.first),
                      LocalModelDependency.create(LocalModelDependencyType.IMPORT, pair.second));
      return true;
    });
  }

  private void collectScanDependentLocalModels(Set<Pair<LocalModel, LocalModelDependency>> models) {
    Module module = getModule();
    List<SpringBeansPackagesScan> scans = getPackagesScans();
    for (SpringBeansPackagesScan packagesScan : scans) {
      if (module.isDisposed()) break;

      Set<CommonSpringBean> beans = packagesScan.getScannedElements(module);
      for (CommonSpringBean bean : beans) {
        if (!(bean instanceof SpringStereotypeElement)) continue;

        SpringStereotypeElement stereotypeElement = (SpringStereotypeElement)bean;
        PsiClass psiClass = stereotypeElement.getPsiElement();
        if (SpringCommonUtils.isSpringBeanCandidateClass(psiClass) &&
            JamService.getJamService(psiClass.getProject())
              .getJamElement(JamPsiMemberSpringBean.PSI_MEMBER_SPRING_BEAN_JAM_KEY, psiClass) != null &&
            SpringProfileUtils.isInActiveProfiles(stereotypeElement, getActiveProfiles())) {
          PsiElement identifyingElementForDependency = packagesScan.getIdentifyingPsiElement();
          if (identifyingElementForDependency == null) {
            String className = psiClass.getQualifiedName();
            if (className != null) {
              Set<PsiPackage> packages = packagesScan.getPsiPackages();
              for (PsiPackage psiPackage : packages) {
                if (psiPackage.containsClassNamed(className)) {
                  identifyingElementForDependency = psiPackage;
                  break;
                }
              }
            }
          }
          if (identifyingElementForDependency == null) {
            identifyingElementForDependency = psiClass;
          }
          LocalModelDependency dependency =
            LocalModelDependency.create(LocalModelDependencyType.COMPONENT_SCAN, identifyingElementForDependency);
          addNotNullModel(models, getLocalAnnotationModel(stereotypeElement.getPsiElement()), dependency);
        }
      }
    }
  }

  @Nullable
  protected LocalAnnotationModel getLocalAnnotationModel(@NotNull PsiClass aClass) {
    return SpringLocalModelFactory.getInstance().getOrCreateLocalAnnotationModel(aClass, getModule(), getActiveProfiles());
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (!(o instanceof LocalAnnotationModel)) return false;

    LocalAnnotationModel model = (LocalAnnotationModel)o;

    if (!myClass.equals(model.myClass)) return false;
    if (!myModule.equals(model.myModule)) return false;
    if (!SpringProfileUtils.profilesAsString(myActiveProfiles).equals(SpringProfileUtils.profilesAsString(model.myActiveProfiles))) {
      return false;
    }


    return true;
  }

  @Override
  public int hashCode() {
    int result = myClass.hashCode();
    result = 31 * result + myModule.hashCode();
    int profilesHashCode = 0;
    for (String profile : myActiveProfiles) {
      if (!profile.equals(SpringProfile.DEFAULT_PROFILE_NAME)) {
        profilesHashCode += profile.hashCode();
      }
    }
    result = 31 * result + profilesHashCode;
    return result;
  }

  @Override
  protected Collection<Object> getCachingProcessorsDependencies() {
    Collection<Object> dependencies = new HashSet<>();

    Collections.addAll(dependencies, SpringModificationTrackersManager.getInstance(myClass.getProject()).getOuterModelsDependencies());
    // all non-libs files in class hierarchy
    Collections.addAll(dependencies, Arrays.stream(myClass.getSupers()).map(aClass -> aClass.getContainingFile())
      .filter(psiFile -> psiFile != null && !(psiFile instanceof ClsFileImpl)).toArray());

    return dependencies;
  }

  @NotNull
  @Override
  public final List<SpringBeanPointer<?>> findQualified(@NotNull SpringQualifier qualifier) {
    return findLocalBeansByQualifier(this, qualifier);  // don't use caching map here (like LocalXmlModel) as found bean could be located in super class.
  }
}