// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.contexts.model;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.ModificationTracker;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiFile;
import com.intellij.psi.PsiType;
import com.intellij.psi.util.CachedValue;
import com.intellij.psi.util.CachedValueProvider;
import com.intellij.psi.util.CachedValuesManager;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.SpringModificationTrackersManager;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.SpringModelSearchParameters;
import com.intellij.spring.model.SpringQualifier;
import com.intellij.util.*;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

//  cache search by name(SpringModelSearchParameters.BeanName) results
public abstract class CacheableCommonSpringModel extends AbstractProcessableModel implements CommonSpringModel {
  private CachedValue<LocalBeansByNameCachingProcessor> myBeanNameCachingProcessor = null;

  public Collection<SpringBeanPointer<?>> getLocalBeans() {return Collections.emptyList();}

  @NotNull
  public Set<String> getProfiles() {
    return Collections.emptySet();
  }

  @NotNull
  @Override
  public Set<String> getActiveProfiles() {
    return Collections.emptySet();
  }

  @Override
  public boolean processByClass(@NotNull SpringModelSearchParameters.BeanClass params,
                                @NotNull Processor<? super SpringBeanPointer<?>> processor) {
    if (!params.canSearch()) return true;
    if (!processLocalBeansByClass(params, processor)) return false;

    return super.processByClass(params, processor);
  }

  public boolean processLocalBeansByClass(@NotNull SpringModelSearchParameters.BeanClass params,
                                          @NotNull Processor<? super SpringBeanPointer<?>> processor) {
    if (!params.canSearch()) return true;
    final PsiType searchType = params.getSearchType();
    if (params.isEffectiveBeanTypes()) {
      for (SpringBeanPointer beanPointer : getLocalBeans()) {
        for (PsiType effectiveBeanType : beanPointer.getEffectiveBeanTypes()) {
          if (!processLocalBeanClass(processor, searchType, beanPointer, effectiveBeanType)) return false;
        }
      }
    }
    else {
      for (SpringBeanPointer beanPointer : getLocalBeans()) {
        if (!processLocalBeanClass(processor, searchType, beanPointer, beanPointer.getSpringBean().getBeanType())) return false;
      }
    }
    return true;
  }

  private static boolean processLocalBeanClass(@NotNull Processor<? super SpringBeanPointer<?>> processor,
                                               @NotNull PsiType searchType,
                                               SpringBeanPointer<?> beanPointer,
                                               @Nullable PsiType beanType) {
    if (beanType != null && searchType.isAssignableFrom(beanType)) {
      return processor.process(beanPointer);
    }
    return true;
  }

  @Override
  public boolean processByName(@NotNull SpringModelSearchParameters.BeanName params,
                               @NotNull Processor<? super SpringBeanPointer<?>> processor) {
    if (!params.canSearch()) return true;
    if (!processLocalBeansByName(params, processor)) return false;

    return super.processByName(params, processor);
  }

  public boolean processLocalBeansByName(@NotNull SpringModelSearchParameters.BeanName params,
                                         @NotNull Processor<? super SpringBeanPointer<?>> processor) {
    if (!params.canSearch()) return true;
    CachedValue<LocalBeansByNameCachingProcessor> cachingProcessor = getBeanNameCachingProcessor();
    return cachingProcessor == null || cachingProcessor.getValue().process(params, processor, getActiveProfiles());
  }

  protected void doProcessLocalBeans(@NotNull SpringModelSearchParameters.BeanName params,
                                     @NotNull Processor<? super SpringBeanPointer<?>> processor) {
    for (SpringBeanPointer<?> beanPointer : getLocalBeans()) {
      if (matchesName(params, beanPointer)) {
        if (!processor.process(beanPointer)) return;
      }
    }
  }

  @Override
  public final boolean processAllBeans(@NotNull Processor<? super SpringBeanPointer<?>> processor) {
    for (SpringBeanPointer<?> pointer : getLocalBeans()) {
      if (!processor.process(pointer)) return false;
    }
    return super.processAllBeans(processor);
  }

  private static boolean matchesName(SpringModelSearchParameters.BeanName params, SpringBeanPointer<?>  pointer) {
    final String paramsBeanName = params.getBeanName();
    if (paramsBeanName.equals(pointer.getName())) return true;

    for (String aliasName : pointer.getAliases()) {
      if (paramsBeanName.equals(aliasName)) return true;
    }

    return false;
  }


  @NotNull
  public Set<String> getAllBeanNames(@NotNull SpringBeanPointer<?>  beanPointer) {
    String beanName = beanPointer.getName();
    if (StringUtil.isEmptyOrSpaces(beanName)) return Collections.emptySet();

    Set<String> names = ContainerUtil.newHashSet(beanName);
    for (String aliasName : beanPointer.getAliases()) {
      if (!StringUtil.isEmptyOrSpaces(aliasName)) names.add(aliasName);
    }
    return names;
  }

  @Nullable
  private CachedValue<LocalBeansByNameCachingProcessor> getBeanNameCachingProcessor() {
    Module module = getModule();
    if (module == null) return null;
    if (myBeanNameCachingProcessor == null) {
      myBeanNameCachingProcessor = CachedValuesManager.getManager(module.getProject()).createCachedValue(() -> {
        return CachedValueProvider.Result.create(new LocalBeansByNameCachingProcessor(), getCachingProcessorsDependencies());
      });
    }
    return myBeanNameCachingProcessor;
  }

  protected Collection<Object> getCachingProcessorsDependencies() {return Collections.singleton(ModificationTracker.EVER_CHANGED);}

  private abstract static class LocalBeansCachingProcessor<InParams extends SpringModelSearchParameters>
    extends SpringCachingProcessor<InParams> {
    @NotNull
    @Override
    protected Collection<SpringBeanPointer<?>> findPointers(@NotNull InParams parameters) {
      final Collection<SpringBeanPointer<?>> results = new SmartList<>();
      Processor<SpringBeanPointer<?>> collectProcessor = Processors.cancelableCollectProcessor(results);

      doProcessBeans(parameters, collectProcessor);

      return results.isEmpty() ? Collections.emptyList() : results;
    }

    @Nullable
    @Override
    protected SpringBeanPointer<?>  findFirstPointer(@NotNull InParams parameters) {
      CommonProcessors.FindFirstProcessor<SpringBeanPointer<?>> firstProcessor = new CommonProcessors.FindFirstProcessor<>();
      doProcessBeans(parameters, firstProcessor);

      return firstProcessor.getFoundValue();
    }

    protected abstract void doProcessBeans(@NotNull InParams parameters, Processor<SpringBeanPointer<?>> collectProcessor);
  }

  private class LocalBeansByNameCachingProcessor extends LocalBeansCachingProcessor<SpringModelSearchParameters.BeanName> {
    @Override
    protected void doProcessBeans(@NotNull SpringModelSearchParameters.BeanName params, Processor<SpringBeanPointer<?>> processor) {
      doProcessLocalBeans(params, processor);
    }
  }

  protected Object @NotNull [] getDependencies(@NotNull Set<PsiFile> containingFiles) {
    final Set<Object> dependencies = new LinkedHashSet<>();

    ContainerUtil.addAllNotNull(dependencies, containingFiles);
    ContainerUtil
      .addAll(dependencies, SpringModificationTrackersManager.getInstance(getModule().getProject()).getOuterModelsDependencies());

    return ArrayUtil.toObjectArray(dependencies);
  }

  @NotNull
  public List<SpringBeanPointer<?>> findQualified(@NotNull final SpringQualifier qualifier) {
    return findLocalBeansByQualifier(this, qualifier);
  }

  protected static List<SpringBeanPointer<?>> findLocalBeansByQualifier(@NotNull CacheableCommonSpringModel model,
                                                                     final SpringQualifier springQualifier) {
    final List<SpringBeanPointer<?>> beans = new SmartList<>();
    for (SpringBeanPointer beanPointer : model.getLocalBeans()) {
      if (!beanPointer.isValid()) continue;
      final CommonSpringBean bean = beanPointer.getSpringBean();
      for (SpringQualifier qualifier : bean.getSpringQualifiers()) {
        if (qualifier.compareQualifiers(springQualifier, model.getModule())) {
          beans.add(beanPointer);
        }
      }
    }
    return beans.isEmpty() ? Collections.emptyList() : beans;
  }
}
