// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.contexts.model;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.AtomicNotNullLazyValue;
import com.intellij.openapi.util.NotNullLazyValue;
import com.intellij.psi.PsiClass;
import com.intellij.spring.model.BeanService;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.jam.stereotype.CustomSpringComponent;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.Set;

public final class BeansSpringModel extends CacheableCommonSpringModel {
  private final NotNullLazyValue<Collection<? extends SpringBeanPointer<?>>> myPointers;
  private final Module myModule;

  public BeansSpringModel(@Nullable Module module, @NotNull final Set<? extends PsiClass> classes) {
    this(module, NotNullLazyValue.lazy(() -> {
      final BeanService beanService = BeanService.getInstance();
      Set<SpringBeanPointer<?>> pointers = new LinkedHashSet<>(classes.size());
      for (PsiClass psiClass : classes) {
        if (psiClass.isValid()) {
          pointers.add(beanService.createSpringBeanPointer(new CustomSpringComponent(psiClass)));
        }
      }
      return pointers;
    }));
  }

  public BeansSpringModel(@Nullable Module module, @NotNull NotNullLazyValue<Collection<? extends SpringBeanPointer<?>>> pointers) {
    assert !(pointers instanceof AtomicNotNullLazyValue);
    myModule = module;
    myPointers = pointers;
  }

  @SuppressWarnings("unchecked")
  @Override
  public Collection<SpringBeanPointer<?>> getLocalBeans() {
    return (Collection<SpringBeanPointer<?>>)myPointers.getValue();
  }

  @Nullable
  @Override
  public Module getModule() {
    return myModule;
  }
}
