// Copyright 2000-2021 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.contexts.model;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.NotNullLazyValue;
import com.intellij.openapi.util.Pair;
import com.intellij.psi.PsiElement;
import com.intellij.psi.util.CachedValue;
import com.intellij.psi.util.CachedValueProvider;
import com.intellij.psi.util.CachedValuesManager;
import com.intellij.spring.SpringModificationTrackersManager;
import com.intellij.spring.contexts.model.graph.LocalModelDependency;
import com.intellij.spring.model.BeanService;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.custom.CustomLocalComponentsDiscoverer;
import com.intellij.util.ArrayUtil;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.stream.Collectors;

public abstract class AbstractSimpleLocalModel<T extends PsiElement> extends CacheableCommonSpringModel implements LocalModel<T> {
  private final NotNullLazyValue<CustomDiscoveredBeansModel> myCustomDiscoveredBeansModel = NotNullLazyValue.volatileLazy(() -> new CustomDiscoveredBeansModel(this));

  @NotNull
  protected CustomDiscoveredBeansModel getCustomDiscoveredBeansModel() {
    return myCustomDiscoveredBeansModel.getValue();
  }

  protected static void addNotNullModel(@NotNull Set<? super Pair<LocalModel, LocalModelDependency>> models,
                                        @Nullable LocalModel model,
                                        @NotNull LocalModelDependency dependency) {
    if (model != null) {
      models.add(Pair.create(model, dependency));
    }
  }

  @Override
  @NotNull
  public Set<LocalModel> getRelatedLocalModels() {
    return getDependentLocalModels().stream().map(pair -> pair.first).filter(model -> !this.equals(model)).collect(Collectors.toSet());
  }

  @Override
  public String toString() {
    return getClass().getSimpleName() + "[" + getConfig() + "]";
  }

  protected static Object @NotNull [] getOutsideModelDependencies(@NotNull LocalModel model) {
    final Project project = model.getConfig().getProject();
    return ArrayUtil.append(SpringModificationTrackersManager.getInstance(project).getOuterModelsDependencies(), model.getConfig());
  }

  protected final static class CustomDiscoveredBeansModel extends CacheableCommonSpringModel {
    @NotNull private final LocalModel<?extends PsiElement> myHostingLocalModel;
    private volatile CachedValue<Collection<SpringBeanPointer<?>>> myCustomDiscoveredBeans;

    CustomDiscoveredBeansModel(@NotNull LocalModel<?extends PsiElement> hostingLocalModel) {
      myHostingLocalModel = hostingLocalModel;
    }

    @Override
    public Collection<SpringBeanPointer<?>> getLocalBeans() {
      Module module = getModule();
      if (module == null) return Collections.emptyList();

      if (myCustomDiscoveredBeans == null) {
        myCustomDiscoveredBeans = CachedValuesManager.getManager(module.getProject()).createCachedValue(() -> {
          Collection<SpringBeanPointer<?>> pointers = computeCustomBeans();

          return CachedValueProvider.Result.create(pointers,
                                                   getDependencies(pointers.stream().map(pointer -> pointer.getContainingFile())
                                                                     .collect(Collectors.toSet())));
        }, false);
      }

      return myCustomDiscoveredBeans.getValue();
    }

    private Collection<SpringBeanPointer<?>> computeCustomBeans() {
      final Set<CommonSpringBean> customSpringComponents = new LinkedHashSet<>();
      for (CustomLocalComponentsDiscoverer discoverer : CustomLocalComponentsDiscoverer.EP_NAME.getExtensionList()) {
        ContainerUtil.addAllNotNull(customSpringComponents, discoverer.getCustomComponents(myHostingLocalModel));
      }
      return BeanService.getInstance().mapSpringBeans(customSpringComponents);
    }

    @Nullable
    @Override
    public Module getModule() {
      return myHostingLocalModel.getModule();
    }
  }
}
