// Copyright 2000-2021 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring;

import com.intellij.ide.TypePresentationService;
import com.intellij.ide.presentation.PresentationProvider;
import com.intellij.openapi.util.NlsSafe;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.pom.PomTargetPsiElement;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiType;
import com.intellij.psi.xml.XmlTag;
import com.intellij.spring.model.*;
import com.intellij.spring.model.jam.JamPsiMemberSpringBean;
import com.intellij.spring.model.jam.JamSpringBeanPointer;
import com.intellij.spring.model.scope.SpringBeanScope;
import com.intellij.spring.model.xml.beans.SpringBean;
import com.intellij.spring.model.xml.beans.SpringImport;
import com.intellij.util.xml.DomUtil;
import com.intellij.util.xml.GenericAttributeValue;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;
import java.util.Objects;

/**
 * @author Yann C&eacute;bron
 */
public class SpringPresentationProvider extends PresentationProvider {

  @Override
  public String getName(Object o) {
    if (o instanceof CommonSpringBean) {
      return getSpringBeanName((CommonSpringBean)o);
    }

    if (o instanceof SpringBeanPointer) {
      return getSpringBeanName((SpringBeanPointer)o);
    }

    if (o instanceof SpringImport) {
      return ((SpringImport)o).getResource().getStringValue();
    }
    return null;
  }

  @Nullable
  @Override
  public Icon getIcon(Object o) {
    return getSpringIcon(o);
  }

  @NotNull
  public static @NlsSafe String getSpringBeanName(SpringBeanPointer springBeanPointer) {
    final String pointerName = springBeanPointer.getName();
    return pointerName != null ? pointerName : getSpringBeanName(springBeanPointer.getSpringBean());
  }

  @NotNull
  public static @NlsSafe String getSpringBeanName(CommonSpringBean springBean) {
    if (!springBean.isValid()) return "<invalid>";

    final String beanName = springBean.getBeanName();
    if (beanName != null) return beanName;

    if (springBean instanceof SpringBean) {
      final PsiType beanClass = springBean.getBeanType();
      if (beanClass != null) return beanClass.getPresentableText();
    }

    final String typeName = TypePresentationService.getService().getTypeName(springBean);
    if (typeName != null) return typeName;

    final PsiElement identifyingPsiElement = springBean.getIdentifyingPsiElement();
    if (identifyingPsiElement instanceof PomTargetPsiElement) {
      final PsiElement target = identifyingPsiElement.getNavigationElement();
      if (target instanceof XmlTag) {
        final String name = ((XmlTag)target).getName();
        if (!StringUtil.isEmptyOrSpaces(name)) return "<" + name + " ... />";
      }
    }

    return SpringApiBundle.message("spring.bean.with.unknown.name");
  }

  public static String getSpringBeanLocation(SpringBeanPointer springBeanPointer) {
    if (springBeanPointer instanceof JamSpringBeanPointer) {
      CommonSpringBean commonSpringBean = springBeanPointer.getSpringBean();
      if (commonSpringBean instanceof SpringImplicitBeanMarker) {
        return ((SpringImplicitBeanMarker)commonSpringBean).getProviderName();
      }
    }

    return springBeanPointer.getContainingFile().getName();
  }

  @Nullable
  public static Icon getSpringIcon(@Nullable Object o) {
    if (o instanceof SpringBeanPointer) {
      o = ((SpringBeanPointer)o).getSpringBean();
    }
    else if (o instanceof SpringBeanPsiTarget) {
      o = ((SpringBeanPsiTarget)o).getSpringBean();
    }

    if (!(o instanceof CommonSpringBean)) {
      return null;
    }

    if (o instanceof SpringImplicitBeanMarker) {
      return SpringApiIcons.ImplicitBean;
    }

    if (o instanceof SpringInfrastructureBean) {
      return SpringApiIcons.InfrastructureBean;
    }

    if (o instanceof JamPsiMemberSpringBean) {
      return SpringApiIcons.SpringJavaBean;
    }

    if (o instanceof SpringBean && ((SpringBean)o).isValid()) {
      SpringBean springBean = (SpringBean)o;

      if (springBean.isAbstract()) {
        return SpringApiIcons.AbtractBean;
      }

      GenericAttributeValue<SpringBeanScope> scope = springBean.getScope();
      if (DomUtil.hasXml(scope) &&
          Objects.equals(scope.getStringValue(), SpringBeanScope.PROTOTYPE_SCOPE.getValue())) {
        return SpringApiIcons.PrototypeBean;
      }
    }

    return SpringApiIcons.SpringBean;
  }
}
