// Copyright 2000-2021 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring;

import com.intellij.openapi.Disposable;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.project.Project;
import com.intellij.psi.PsiFile;
import com.intellij.spring.contexts.model.SpringModel;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Set;

/**
 * Provides access to Spring models and their dependencies.
 */
public abstract class SpringManager implements Disposable {

  public static SpringManager getInstance(Project project) {
    return project.getService(SpringManager.class);
  }

  public abstract Object @NotNull [] getModelsDependencies(@NotNull Module module, Object... additional);

  // scans all modules in project. returns SpringModels or AnnotationSpringModel(if xml files are added as @ImportResource).
  // if file wasn't added in any fileset then create local model for current file (LocalXmlModel or LocalAnnotationModel)
  @NotNull
  public abstract Set<SpringModel> getSpringModelsByFile(@NotNull PsiFile file);

  @Nullable
  public abstract SpringModel getSpringModelByFile(@NotNull PsiFile file);

  /**
   * Returns all models configured in given module including all dependencies.
   *
   * @param module a module.
   * @return Models or empty set if none configured.
   * @see #getCombinedModel(Module)
   */
  @NotNull
  public abstract Set<SpringModel> getAllModels(@NotNull Module module);

  /**
   * Returns all models configured in given module without dependencies.
   * <p/>
   * Usually, you will want to use {@link #getAllModels(Module)}.
   *
   * @param module Module.
   * @return Models or empty set if none configured.
   */
  public abstract Set<SpringModel> getAllModelsWithoutDependencies(@NotNull Module module);

  /**
   * Returns result of merging all models configured in given module.
   *
   * @param module a module.
   * @return Merged model.
   * @see #getAllModels(Module)
   */
  @NotNull
  public abstract SpringModel getCombinedModel(@NotNull Module module);
}
