/*
 * Copyright 2000-2018 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring;

import com.intellij.jam.JavaLibraryUtils;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.roots.ProjectRootManager;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.JavaPsiFacade;
import com.intellij.psi.PsiClass;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.psi.util.CachedValueProvider.Result;
import com.intellij.psi.util.CachedValuesManager;
import com.intellij.spring.constants.SpringConstants;
import com.intellij.util.ArrayUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import static com.intellij.spring.SpringLibraryUtil.SpringVersion.ANY;

public final class SpringLibraryUtil {

  public enum SpringVersion {
    ANY("1.0", SpringConstants.BEAN_FACTORY_CLASS),
    V_2_5("2.5", "org.springframework.core.PriorityOrdered"),
    V_4_2("4.2", "org.springframework.beans.AbstractNestablePropertyAccessor"),
    V_4_3("4.3", "org.springframework.core.MethodClassKey"),

    V_5_0("5.0", "org.springframework.core.ReactiveAdapter"),

    V_5_1("5.1", "org.springframework.core.codec.Hints"),

    V_5_2("5.2", "org.springframework.core.SortedProperties");

    private final String myVersion;
    private final String myDetectionClassFqn;

    SpringVersion(String version, String detectionClassFqn) {
      myVersion = version;
      myDetectionClassFqn = detectionClassFqn;
    }

    boolean isAtLeast(SpringVersion reference) {
      if (reference == ANY) {
        return true;
      }

      return StringUtil.compareVersionNumbers(getVersion(), reference.getVersion()) >= 0;
    }

    String getVersion() {
      return myVersion;
    }

    String getDetectionClassFqn() {
      return myDetectionClassFqn;
    }
  }

  public static boolean hasSpringLibrary(@NotNull Project project) {
    return JavaLibraryUtils.hasLibraryClass(project, ANY.getDetectionClassFqn());
  }

  public static boolean hasSpringLibrary(@Nullable Module module) {
    return isAtLeastVersion(module, ANY);
  }

  public static boolean isAtLeastVersion(@Nullable Module module, SpringVersion version) {
    if (module == null) return false;

    if (!hasSpringLibrary(module.getProject())) return false;

    final SpringVersion cached = getCachedSpringVersion(module);
    return cached != null && cached.isAtLeast(version);
  }

  @Nullable
  private static SpringVersion getCachedSpringVersion(@NotNull final Module module) {
    final Project project = module.getProject();

    return CachedValuesManager.getManager(project).getCachedValue(module, () -> {
      final GlobalSearchScope scope = GlobalSearchScope.moduleRuntimeScope(module, false);
      final JavaPsiFacade javaPsiFacade = JavaPsiFacade.getInstance(project);

      SpringVersion detected = null;
      for (SpringVersion version : ArrayUtil.reverseArray(SpringVersion.values())) {
        final PsiClass psiClass = javaPsiFacade.findClass(version.getDetectionClassFqn(), scope);
        if (psiClass != null) {
          detected = version;
          break;
        }
      }

      return Result.create(detected, ProjectRootManager.getInstance(project));
    });
  }
}
