// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring;

import com.intellij.codeHighlighting.HighlightDisplayLevel;
import com.intellij.codeInspection.LocalInspectionTool;
import com.intellij.codeInspection.ProblemDescriptor;
import com.intellij.openapi.application.ApplicationManager;
import com.intellij.openapi.compiler.CompileContext;
import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.psi.PsiFile;
import org.jetbrains.annotations.TestOnly;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;

/**
 * Registry of all configuration file inspections provided by Spring-* plugins.
 *
 * @author Yann C&eacute;bron
 */
public abstract class SpringInspectionsRegistry {

  /**
   * Provide plugin-specific configuration file inspections.
   */
  public interface Contributor {

    ExtensionPointName<SpringInspectionsRegistry.Contributor> EP_NAME =
      ExtensionPointName.create("com.intellij.spring.inspectionsRegistryContributor");

    Class<? extends LocalInspectionTool>[] getInspectionClasses();
  }

  /**
   * Allows adding additional configuration files as well as additional highlighting to Spring validation context.
   */
  public abstract static class AdditionalFilesContributor {

    public static final ExtensionPointName<SpringInspectionsRegistry.AdditionalFilesContributor> EP_NAME =
      ExtensionPointName.create("com.intellij.spring.inspectionsRegistryAdditionalFilesContributor");

    public abstract Collection<VirtualFile> getAdditionalFilesToProcess(final Project project, final CompileContext context);

    /**
     * Add additional highlighting (e.g. provided by Annotators) for the given file.
     * <p/>
     * NOTE: all files from current validation context will be passed in, not only additional ones provided by this EP.
     *
     * @param psiFile File to check.
     * @return Additional highlighting info.
     */
    public Map<ProblemDescriptor, HighlightDisplayLevel> checkAdditionally(PsiFile psiFile) {
      return Collections.emptyMap();
    }
  }

  public static SpringInspectionsRegistry getInstance() {
    return ApplicationManager.getApplication().getService(SpringInspectionsRegistry.class);
  }

  /**
   * Returns all registered inspections. Used by Compiler|Validation "Spring Model".
   *
   * @return Inspections.
   */
  public abstract Class<? extends LocalInspectionTool>[] getSpringInspectionClasses();

  @TestOnly
  public abstract Class<? extends LocalInspectionTool>[] getTestSpringInspectionClasses();

  @TestOnly
  public abstract Class<? extends LocalInspectionTool> getTestSpringModelInspectionClass();
}
