// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring;

import com.intellij.openapi.module.Module;
import com.intellij.spring.contexts.model.visitors.CommonSpringModelVisitorContext;
import com.intellij.spring.contexts.model.visitors.SpringModelVisitors;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.SpringModelSearchParameters;
import com.intellij.util.Processor;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;
import java.util.Set;

public interface CommonSpringModel {

  /**
   * Spring models:
   * - have connections/links/associations to other models (via imports/component-scans/implicit fileset configurations/auto-configurations/etc.)
   * - contain other models {@link com.intellij.spring.contexts.model.SpringModel}
   *
   * @return related models
   * @see SpringModelVisitors#visitRelatedModels(CommonSpringModel, CommonSpringModelVisitorContext)
   */
  default Set<CommonSpringModel> getRelatedModels() {
    return Collections.emptySet();
  }

  /**
   * @param params    Search parameters.
   * @param processor Results processor.
   * @return {@code true} to continue processing or {@code false} to stop.
   * @see com.intellij.spring.model.utils.SpringModelSearchers
   * @see SpringModelSearchParameters#canSearch()
   */
  boolean processByClass(@NotNull SpringModelSearchParameters.BeanClass params, @NotNull Processor<? super SpringBeanPointer<?>> processor);

  /**
   * @param params    Search parameters.
   * @param processor Results processor.
   * @return {@code true} to continue processing or {@code false} to stop.
   * @see com.intellij.spring.model.utils.SpringModelSearchers
   * @see SpringModelSearchParameters#canSearch()
   */
  boolean processByName(@NotNull SpringModelSearchParameters.BeanName params, @NotNull Processor<? super SpringBeanPointer<?>> processor);

  /**
   * @param processor Results processor.
   * @return {@code true} to continue processing or {@code false} to stop.
   * @see com.intellij.spring.model.utils.SpringModelSearchers
   */
  default boolean processAllBeans(@NotNull Processor<? super SpringBeanPointer<?>> processor) {return true;}

  /**
   * NOTE: Expensive operation. Consider using {@code process...()} methods instead.
   *
   * @return All beans.
   */
  @NotNull
  Collection<SpringBeanPointer<?>> getAllCommonBeans();

  /**
   * Returns the associated module.
   *
   * @return Module containing this model or {@code null} if model is not bound to a specific module (e.g. Project-global).
   */
  @Nullable
  Module getModule();

  @Nullable
  Set<String> getActiveProfiles();
}
