/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.javaee.web.model.jam;

import com.intellij.jam.JamElement;
import com.intellij.jam.JamStringAttributeElement;
import com.intellij.jam.annotations.JamPsiValidity;
import com.intellij.jam.model.common.CommonModelElement;
import com.intellij.jam.reflect.*;
import com.intellij.javaee.utils.JavaeeType;
import com.intellij.javaee.web.CommonServlet;
import com.intellij.javaee.web.CommonServletMapping;
import com.intellij.psi.PsiAnnotation;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiElementRef;
import com.intellij.psi.ref.AnnotationChildLink;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.xml.NameValue;
import org.jetbrains.annotations.NotNull;

import java.util.List;

import static com.intellij.javaee.web.WebCommonClassNames.ANNOTATION_WEB_SERVLET;

/**
 * @author peter
 */
public abstract class JamServlet extends CommonModelElement.PsiBase implements CommonServlet, CommonServletMapping<CommonServlet>, JamElement {

  public static final JamStringAttributeMeta.Single<String> NAME_META = JamAttributeMeta.singleString("name");
  public static final JamStringAttributeMeta.Collection<String> URL_PATTERNS_META = JamAttributeMeta.collectionString("urlPatterns");
  public static final JamStringAttributeMeta.Collection<String> VALUES_META = JamAttributeMeta.collectionString("value");

  public static final JamAnnotationAttributeMeta.Collection<JamInitParam> JAVAX_INIT_PARAMS_META =
    JamAttributeMeta.annoCollection("initParams", JamInitParam.JAVAX_INIT_PARAM_ANNO_META, JamInitParam.class);

  public static final JamAnnotationAttributeMeta.Collection<JamInitParam> JAKARTA_INIT_PARAMS_META =
    JamAttributeMeta.annoCollection("initParams", JamInitParam.JAKARTA_INIT_PARAM_ANNO_META, JamInitParam.class);

  public static final JamAnnotationArchetype ANNOTATION_ARCHETYPE = new JamAnnotationArchetype().
    addAttribute(NAME_META).
    addAttribute(URL_PATTERNS_META).
    addAttribute(VALUES_META);

  public static final JamClassMeta<JamServlet> SERVLET_CLASS_META = new JamClassMeta<>(JamServlet.class)
    .addAnnotation(new JamAnnotationMeta(ANNOTATION_WEB_SERVLET.javax(), ANNOTATION_ARCHETYPE).addAttribute(JAVAX_INIT_PARAMS_META))
    .addAnnotation(new JamAnnotationMeta(ANNOTATION_WEB_SERVLET.jakarta(), ANNOTATION_ARCHETYPE).addAttribute(JAKARTA_INIT_PARAMS_META));

  private final PsiClass myPsiClass;
  private final PsiElementRef<PsiAnnotation> myAnno;
  private final @NotNull JavaeeType myJavaeeType;

  @Override
  @NotNull
  public PsiClass getPsiClass() {
    return myPsiClass;
  }

  protected JamServlet(PsiClass aClass) {
    myPsiClass = aClass;
    myJavaeeType = ANNOTATION_WEB_SERVLET.typeFromAnnotated(aClass);
    myAnno = AnnotationChildLink.createRef(aClass, ANNOTATION_WEB_SERVLET.fqn(myJavaeeType));
  }

  @NotNull
  @Override
  public PsiElement getPsiElement() {
    return getPsiClass();
  }

  @Override
  @JamPsiValidity
  public abstract boolean isValid();

  @Override
  @NameValue
  public JamStringAttributeElement<String> getServletName() {
    return NAME_META.getJam(myAnno);
  }

  @Override
  public List<JamStringAttributeElement<String>> getUrlPatterns() {
    return ContainerUtil.concat(URL_PATTERNS_META.getJam(myAnno), VALUES_META.getJam(myAnno));
  }

  @Override
  public List<JamInitParam> getInitParams() {
    return myJavaeeType.equals(JavaeeType.JAVAX) ? JAVAX_INIT_PARAMS_META.getJam(myAnno) : JAKARTA_INIT_PARAMS_META.getJam(myAnno);
  }

  @Override
  public JamInitParam addInitParam() {
    return myJavaeeType.equals(JavaeeType.JAVAX) ?
     JAVAX_INIT_PARAMS_META.addAttribute(PsiElementRef.real(myAnno.getPsiElement()))
                                          : JAKARTA_INIT_PARAMS_META.addAttribute(PsiElementRef.real(myAnno.getPsiElement()));
  }

  @Override
  public CommonServlet getServlet() {
    return this;
  }

  @Override
  public PsiElement getMappingElement() {
    return myAnno.getPsiElement();
  }
}
