/*
 * Copyright 2000-2007 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.javaee.web.facet;

import com.intellij.facet.*;
import com.intellij.javaee.facet.JavaeeFacet;
import com.intellij.javaee.web.WebDirectoryElement;
import com.intellij.javaee.web.WebModel;
import com.intellij.javaee.web.WebRoot;
import com.intellij.javaee.web.WebUtil;
import com.intellij.javaee.web.model.xml.WebApp;
import com.intellij.javaee.web.model.xml.WebAppVersion;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.psi.PsiDirectory;
import com.intellij.util.descriptors.ConfigFile;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.List;
import java.util.Map;

public abstract class WebFacet extends JavaeeFacet {
  public static final FacetTypeId<WebFacet> ID = new FacetTypeId<>("web");

  protected WebFacet(@NotNull FacetType facetType,
                     @NotNull Module module,
                     String name,
                     @NotNull FacetConfiguration configuration,
                     Facet underlyingFacet) {
    super(facetType, module, name, configuration, underlyingFacet);
  }

  public static @NotNull Collection<WebFacet> getInstances(@NotNull Module module) {
    return FacetManager.getInstance(module).getFacetsByType(ID);
  }

  public abstract WebRoot addWebRoot(@NotNull String directoryUrl, @NotNull String relativePath);

  public abstract WebRoot addWebRoot(@NotNull VirtualFile directory, @NotNull String relativePath);

  public abstract void removeWebRoot(WebRoot webRoot);

  public abstract void removeAllWebRoots();

  public abstract WebRoot addWebRootNoFire(@NotNull String directoryUrl, @NotNull String relativePath);

  public abstract List<WebRoot> getWebRoots();

  public abstract List<WebRoot> getWebRoots(boolean includeDependentModules);

  /**
   * Merged model for web.xml and all web fragments.
   * Should be used for read access only.
   * @see #getWebApps()
   */
  public abstract @Nullable WebApp getRoot();

  public abstract List<WebApp> getWebApps();

  public abstract @NotNull WebAppVersion getWebAppVersion();

  public abstract @NotNull WebModel getWebModel();

  public abstract List<PsiDirectory> getResources();

  public abstract Map<String, String> getTaglibUriToResourceMap();

  public abstract @Nullable ConfigFile getWebXmlDescriptor();

  public abstract void addWebSourceRoot(@NotNull String url);
  public abstract void removeWebSourceRoot(@NotNull String url);
  public abstract void setWebSourceRoots(String @NotNull [] urls);

  public abstract List<String> getWebSourceRootUrls();

  public abstract Collection<VirtualFile> getSourceRoots();

  public static @Nullable WebDirectoryElement findWebDirectoryElement(@Nullable Module module, @NotNull String path) {
    if (module == null) {
      return null;
    }
    for (WebFacet webFacet : getInstances(module)) {
      WebDirectoryElement element = WebUtil.getWebUtil().findWebDirectoryElement(path, webFacet);
      if (element != null) return element;
    }
    return null;
  }
}
