package com.intellij.javaee.utils;

import com.intellij.codeInsight.AnnotationUtil;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.project.Project;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiModifierListOwner;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class JavaeeClass {
  @NotNull private final String myJavaxFqn;
  @NotNull private final String myJakartaFqn;

  protected JavaeeClass(@NotNull String javaxFqn) {
    if (javaxFqn.startsWith(JavaeeType.JAVAX.pkg)) {
      myJavaxFqn = javaxFqn;
      myJakartaFqn = javaxFqn.replace(JavaeeType.JAVAX.pkg, JavaeeType.JAKARTA.pkg);
    } else {
      myJakartaFqn = javaxFqn;
      myJavaxFqn = javaxFqn.replace(JavaeeType.JAKARTA.pkg, JavaeeType.JAVAX.pkg);
    }
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (!(o instanceof JavaeeClass)) return false;

    JavaeeClass aClass = (JavaeeClass)o;

    if (!myJavaxFqn.equals(aClass.myJavaxFqn)) return false;

    return true;
  }

  @Override
  public int hashCode() {
    return myJavaxFqn.hashCode();
  }

  @Override
  public String toString() {
    return javax();
  }

  public static JavaeeClass create(@NotNull String fqn) {
    return new JavaeeClass(fqn);
  }

  public String fqn(@NotNull Module module) {
    return fqn(JavaeeType.discover(module, this));
  }

  @NotNull
  public String fqn(@NotNull Project project) {
    return fqn(JavaeeType.discover(project));
  }

  public @NotNull String fqn(@NotNull PsiElement context) {
    return fqn(JavaeeType.discover(context, this));
  }

  public @NotNull String fqnFromAnnotated(@NotNull PsiModifierListOwner context) {
    if (AnnotationUtil.isAnnotated(context, javax(), 0)) {
      return javax();
    }
    if (AnnotationUtil.isAnnotated(context, jakarta(), 0)) {
      return jakarta();
    }
    return fqn(context);
  }

  public @NotNull JavaeeType typeFromAnnotated(@NotNull PsiModifierListOwner context) {
    if (AnnotationUtil.isAnnotated(context, javax(), 0)) {
      return JavaeeType.JAVAX;
    }
    if (AnnotationUtil.isAnnotated(context, jakarta(), 0)) {
      return JavaeeType.JAKARTA;
    }
    return JavaeeType.discover(context, this);
  }

  public boolean isAnnotated(@NotNull PsiModifierListOwner listOwner, int flags) {
    return AnnotationUtil.isAnnotated(listOwner, javax(), flags) || AnnotationUtil.isAnnotated(listOwner, jakarta(), flags);
  }

  @NotNull
  public String fqn(@NotNull JavaeeType javaeeType) {
    return JavaeeType.JAVAX.equals(javaeeType) ? myJavaxFqn : myJakartaFqn;
  }

  @NotNull
  public String javax() { return myJavaxFqn; }

  @NotNull
  public String jakarta() { return myJakartaFqn; }

  public String[] all() {
    return new String[]{myJavaxFqn, myJakartaFqn};
  }

  public boolean isFqn(@Nullable String fqn) {
    return myJavaxFqn.equals(fqn) || myJakartaFqn.equals(fqn);
  }
}
