/*
 * Copyright 2000-2007 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.javaee.ui;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.NlsSafe;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiDirectory;
import com.intellij.psi.codeStyle.CodeStyleSettingsManager;
import com.intellij.psi.util.ClassUtil;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;

import java.util.Collection;
import java.util.List;
import java.util.Objects;

public abstract class NewAbstractDataHolder {
  private String myName;
  private String myClassName;
  private PsiDirectory myClassDirectory;
  protected final JavaeeCodeStyleSettings myCodeStyleSettings;
  protected final Module myModule;
  private final List<?> myExistings;
  private final String myOldName;

  protected NewAbstractDataHolder(@NotNull Module module, List<?> existings, String oldName) {
    myModule = module;
    myExistings = existings;
    myOldName = oldName;
    myCodeStyleSettings = CodeStyleSettingsManager.getSettings(module.getProject()).getCustomSettings(JavaeeCodeStyleSettings.class);
  }

  public Module getModule() {
    return myModule;
  }

  @NonNls protected abstract String getDefaultName();

  protected abstract String getObjectName(Object o);

  protected String getUniqueName() {
    final Collection<String> existingNames = ContainerUtil.map2Set(myExistings, s -> getObjectName(s));

    String defaultName = getDefaultName();
    if (!existingNames.contains(getFullName(defaultName))) {
      return defaultName;
    }

    for (int i = 2; ; i++) {
      String name = defaultName + i;
      if (!existingNames.contains(getFullName(name))) {
        return name;
      }
    }
  }

  private String getFullName(String defaultName) {
    return getNamePrefix(myCodeStyleSettings) + defaultName + getNameSuffix(myCodeStyleSettings);
  }

  public abstract String getNamePrefix(JavaeeCodeStyleSettings settings);

  public abstract String getNameSuffix(JavaeeCodeStyleSettings settings);

  public String getName() {
    return myName;
  }

  public void setName(@NonNls String name) {
    myName = name;
  }

  @NlsSafe
  public String getClassName() {
    return myClassName;
  }

  public void setClassName(@NonNls String className) {
    myClassName = className;
  }

  public PsiDirectory getClassDirectory() {
    return myClassDirectory;
  }

  public void setClassDirectory(PsiDirectory classDirectory) {
    myClassDirectory = ClassUtil.sourceRoot(classDirectory);
  }

  public void setPackage(String packageName) {
    myClassName = ClassUtil.createNewClassQualifiedName(packageName, myClassName);
  }

  @NlsSafe
  public String getPackage() {
    return ClassUtil.extractPackageName(getClassName());
  }

  public String getPureName() {
    String result = getName();
    String namePrefix = getNamePrefix(myCodeStyleSettings);
    result = StringUtil.trimStart(result, namePrefix);
    String nameSuffix = getNameSuffix(myCodeStyleSettings);
    result = StringUtil.trimEnd(result, nameSuffix);
    return result;
  }

  public boolean checkNameIsUnique() {
    String name = getName();
    if (Objects.equals(myOldName, name)) {
      return true;
    }

    for (final Object existing : myExistings) {
      String anotherName = getObjectName(existing);
      if (Objects.equals(anotherName, myOldName)) {
        continue;
      }
      if (Objects.equals(anotherName, name)) {
        return false;
      }

    }
    return true;
  }


}
