package com.intellij.javaee;

import com.intellij.framework.FrameworkAvailabilityCondition;
import com.intellij.framework.FrameworkVersion;
import com.intellij.ide.util.frameworkSupport.FrameworkSupportModel;
import com.intellij.javaee.utils.JavaeeType;
import org.jetbrains.annotations.NotNull;

/**
 * @author Gregory.Shrago
 */
public enum JavaeeVersion implements FrameworkVersion {
  J2EE_1_4("1.4", "J2EE 1.4"),
  JAVAEE_5("5", "Java EE 5"),
  JAVAEE_6("6", "Java EE 6"),
  JAVAEE_7("7", "Java EE 7"),
  JAVAEE_8("8", "Java EE 8"), // the latest Java EE supported in Add Frameworks Support dialog
  JAKARTA_9("9", "Jakarta EE 9", true);

  private final String myId;
  private final String myPresentableName;
  private final boolean myRequiresJakarta;

  JavaeeVersion(@NotNull String id, @NotNull String presentableName) {
    myId = id;
    myPresentableName = presentableName;
    myRequiresJakarta = false;
  }

  JavaeeVersion(String id, String presentableName, boolean requiresJakarta) {
    myId = id;
    myPresentableName = presentableName;
    myRequiresJakarta = requiresJakarta;
  }

  @NotNull
  @Override
  public String getId() {
    return myId;
  }

  @NotNull
  @Override
  public String getPresentableName() {
    return myPresentableName;
  }

  @Override
  public String getVersionNumber() {
    return getId();
  }

  @NotNull
  @Override
  public FrameworkAvailabilityCondition getAvailabilityCondition() {
    if (myRequiresJakarta) {
      // Jakarta EE projects must be created using Maven/Gradle and not JPS
      return new JakartaFrameworkAvailabilityCondition();
    }
    return FrameworkAvailabilityCondition.ALWAYS_TRUE;
  }

  private static class JakartaFrameworkAvailabilityCondition extends FrameworkAvailabilityCondition {
    @Override
    public boolean isAvailableFor(@NotNull FrameworkSupportModel model) {
      if (model.getProject() != null && model.getModuleBuilder() == null) {
        return JavaeeType.discover(model.getProject()) == JavaeeType.JAKARTA;
      }
      return false;
    }
  }
}
