package com.intellij.psi.css.impl.util.table;

import com.intellij.codeInspection.LocalQuickFix;
import com.intellij.css.util.CssPsiUtil;
import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.NlsSafe;
import com.intellij.openapi.util.TextRange;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiReference;
import com.intellij.psi.css.*;
import com.intellij.psi.css.descriptor.*;
import com.intellij.psi.css.descriptor.value.CssValueDescriptor;
import com.intellij.psi.css.descriptor.value.CssValueValidator;
import com.intellij.psi.css.descriptor.value.CssValueValidatorStub;
import com.intellij.psi.css.resolve.CssStyleReferenceStub;
import com.intellij.psi.util.*;
import com.intellij.util.ArrayUtil;
import com.intellij.util.ArrayUtilRt;
import com.intellij.util.NotNullFunction;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.awt.*;
import java.util.List;
import java.util.*;

public final class CssDescriptorsUtil {

  public static final NotNullFunction<CssElementDescriptor, String> GET_DESCRIPTOR_ID_FUNCTION = CssElementDescriptor::getId;

  private static final Key<ParameterizedCachedValue<CssElementDescriptorProvider, PsiElement>> DESCRIPTOR_PROVIDER_KEY =
    Key.create("css.descriptor.provider");
  private static final ParameterizedCachedValueProvider<CssElementDescriptorProvider, PsiElement> DESCRIPTOR_PROVIDER_CACHE_VALUE =
    param -> CachedValueProvider.Result.create(innerFindDescriptorProvider(param), param);

  private CssDescriptorsUtil() {
  }

  public static @Nullable CssElementDescriptorProvider findDescriptorProvider(final @Nullable PsiElement context) {
    final CssStylesheet stylesheet = PsiTreeUtil.getNonStrictParentOfType(context, CssStylesheet.class);
    if (stylesheet != null) {
      return CachedValuesManager.getManager(stylesheet.getProject()).getParameterizedCachedValue(stylesheet, DESCRIPTOR_PROVIDER_KEY,
                                                                                                 DESCRIPTOR_PROVIDER_CACHE_VALUE, false,
                                                                                                 context);
    }
    return innerFindDescriptorProvider(context);
  }

  private static CssElementDescriptorProvider innerFindDescriptorProvider(@Nullable PsiElement context) {
    List<CssElementDescriptorProvider> applicableProviders = new LinkedList<>();
    CssElementDescriptorProvider[] providers = CssElementDescriptorProvider.EP_NAME.getExtensions();
    for (CssElementDescriptorProvider provider : providers) {
      if (provider.isMyContext(context)) {
        applicableProviders.add(provider);

        if (!provider.shouldAskOtherProviders(context)) {
          break;
        }
      }
    }
    if (applicableProviders.size() == 1) {
      return ContainerUtil.getFirstItem(applicableProviders);
    }
    return !applicableProviders.isEmpty() ? new CompositeCssElementDescriptorProvider(applicableProviders) : null;
  }

  public static String @NotNull [] getSimpleSelectors(@NotNull PsiElement context) {
    CssElementDescriptorProvider provider = findDescriptorProvider(context);
    return provider != null ? provider.getSimpleSelectors(context) : ArrayUtilRt.EMPTY_STRING_ARRAY;
  }

  /**
   * @deprecated use {@link #getPropertyDescriptors(String, PsiElement)}
   */
  @Deprecated
  public static @Nullable CssPropertyDescriptor getPropertyDescriptor(@NotNull CssDeclaration declaration) {
    return getPropertyDescriptor(declaration.getPropertyName(), declaration);
  }

  public static @NotNull Collection<? extends CssPropertyDescriptor> getAllPropertyDescriptors(@Nullable PsiElement context) {
    CssElementDescriptorProvider provider = findDescriptorProvider(context);
    return provider != null ? provider.getAllPropertyDescriptors(context) : Collections.emptyList();
  }

  public static @NotNull Collection<? extends CssPseudoSelectorDescriptor> getPseudoSelectorDescriptors(@NotNull String pseudoSelectorName,
                                                                                                        @Nullable PsiElement context) {
    CssElementDescriptorProvider provider = findDescriptorProvider(context);
    return provider != null
           ? provider.findPseudoSelectorDescriptors(pseudoSelectorName, context)
           : Collections.emptyList();
  }

  public static @NotNull Collection<? extends CssFunctionDescriptor> getFunctionDescriptors(@NotNull String functionName,
                                                                                            @Nullable PsiElement context) {
    CssElementDescriptorProvider provider = findDescriptorProvider(context);
    return provider != null
           ? provider.findFunctionDescriptors(functionName, context)
           : Collections.emptyList();
  }

  public static @NotNull Collection<? extends CssPropertyDescriptor> getPropertyDescriptors(@NotNull String propertyName,
                                                                                            @Nullable PsiElement context) {
    CssElementDescriptorProvider provider = findDescriptorProvider(context);
    return provider != null
           ? provider.findPropertyDescriptors(propertyName, context)
           : Collections.emptyList();
  }

  public static @NotNull Collection<? extends CssMediaFeatureDescriptor> getMediaFeatureDescriptors(@NotNull String featureName,
                                                                                                    @Nullable PsiElement context) {
    CssElementDescriptorProvider provider = findDescriptorProvider(context);
    return provider != null
           ? provider.findMediaFeatureDescriptors(featureName, context)
           : Collections.emptyList();
  }

  public static @NotNull Collection<? extends CssMediaFeatureDescriptor> getAllMediaFeatureDescriptors(@Nullable PsiElement context) {
    CssElementDescriptorProvider provider = findDescriptorProvider(context);
    return provider != null ? provider.getAllMediaFeatureDescriptors(context) : Collections.emptyList();
  }

  /**
   * @deprecated use {@link this#getPropertyDescriptors(String, PsiElement)}
   */
  @Deprecated
  public static @Nullable CssPropertyDescriptor getPropertyDescriptor(@NotNull String propertyName, @Nullable PsiElement context) {
    CssElementDescriptorProvider provider = findDescriptorProvider(context);
    return provider != null ? provider.getPropertyDescriptor(propertyName, context) : null;
  }

  /**
   * @return true if given string is possible simple selector name in given context.
   * <p>
   * Nullable context means that context is simple ruleset.
   */
  public static boolean isPossibleSelector(@NotNull String selector, @NotNull PsiElement context) {
    CssElementDescriptorProvider provider = findDescriptorProvider(context);
    return provider == null || provider.isPossibleSelector(selector, context);
  }

  public static @NotNull Collection<LocalQuickFix> getQuickFixesForUnknownSimpleSelector(@NotNull String selectorName,
                                                                                         @NotNull PsiElement context,
                                                                                         boolean isOnTheFly) {
    final Set<LocalQuickFix> result = new HashSet<>();
    CssElementDescriptorProvider[] providers = CssElementDescriptorProvider.EP_NAME.getExtensions();
    for (CssElementDescriptorProvider provider : providers) {
      Collections.addAll(result, provider.getQuickFixesForUnknownSimpleSelector(selectorName, context, isOnTheFly));
    }
    return result;
  }

  public static @NotNull Collection<LocalQuickFix> getQuickFixesForUnknownProperty(@NotNull String propertyName,
                                                                                   @NotNull PsiElement context,
                                                                                   boolean isOnTheFly) {
    final Set<LocalQuickFix> result = new HashSet<>();
    CssElementDescriptorProvider[] providers = CssElementDescriptorProvider.EP_NAME.getExtensions();
    for (CssElementDescriptorProvider provider : providers) {
      Collections.addAll(result, provider.getQuickFixesForUnknownProperty(propertyName, context, isOnTheFly));
    }
    return result;
  }


  /**
   * Retrieves list of pseudo-classes and pseudo-elements descriptors, that allowed in given context.
   * Uses in completion providers.
   *
   * @param context Given context. If {@code null} then all possible variants will be returned.
   */
  public static @NotNull Collection<? extends CssPseudoSelectorDescriptor> getAllPseudoSelectorDescriptors(@Nullable PsiElement context) {
    CssElementDescriptorProvider provider = findDescriptorProvider(context);
    return provider != null ? provider.getAllPseudoSelectorDescriptors(context) : Collections.<CssPseudoClassDescriptor>emptyList();
  }

  public static @NotNull @NlsSafe String getCanonicalPropertyName(@NotNull CssDeclaration declaration) {
    String propertyName = declaration.getPropertyName();
    CssPropertyDescriptor descriptor = getPropertyDescriptor(propertyName, declaration);
    return descriptor != null ? descriptor.toCanonicalName(propertyName) : propertyName;
  }

  public static @Nullable <T extends CssElementDescriptor> T getDescriptorFromLatestSpec(Collection<T> descriptors) {
    T result = null;
    for (T descriptor : descriptors) {
      if (result == null || descriptor.getCssVersion().value() >= result.getCssVersion().value()) {
        result = descriptor;
      }
    }
    return result;
  }

  public static String @NotNull [] extractDescriptorsIdsAsArray(Collection<? extends CssElementDescriptor> descriptors) {
    return ContainerUtil.map2Array(descriptors, String.class, GET_DESCRIPTOR_ID_FUNCTION);
  }

  public static @NotNull <T extends CssMediaGroupAwareDescriptor> Collection<T> filterDescriptorsByMediaType(@NotNull Collection<T> descriptors,
                                                                                                             @Nullable PsiElement context) {
    final Set<CssMediaType> allowedMediaTypes = CssPsiUtil.getAllowedMediaTypesInContext(context);
    if (allowedMediaTypes.contains(CssMediaType.ALL)) {
      return descriptors;
    }

    boolean insidePageRule = false;
    CssAtRule atRule = PsiTreeUtil.getNonStrictParentOfType(context, CssAtRule.class);
    while (atRule != null) {
      if (atRule.getType() == CssContextType.PAGE || atRule.getType() == CssContextType.PAGE_MARGIN) {
        insidePageRule = true;
        break;
      }
      atRule = PsiTreeUtil.getParentOfType(atRule, CssAtRule.class);
    }

    final boolean finalInsidePageRule = insidePageRule;
    return ContainerUtil.filter(descriptors, t -> {
      for (CssMediaType mediaType : allowedMediaTypes) {
        CssMediaGroup[] propertyMediaGroups = t.getMediaGroups();
        for (CssMediaGroup propertyMediaGroup : propertyMediaGroups) {
          if (propertyMediaGroup == CssMediaGroup.ALL
              || finalInsidePageRule && propertyMediaGroup == CssMediaGroup.PAGED
              || ArrayUtil.contains(propertyMediaGroup, (Object[])mediaType.getSupportedGroups())) {
            return true;
          }
        }
      }
      return false;
    });
  }

  public static @NotNull <T extends CssElementDescriptor> Collection<T> filterDescriptorsByContext(@NotNull Collection<T> descriptors,
                                                                                                   @Nullable PsiElement context) {
    CssElementDescriptorProvider provider = findDescriptorProvider(context);
    final CssContextType ruleType = provider != null ? provider.getCssContextType(context) : CssContextType.ANY;

    if (ruleType == CssContextType.ANY) {
      return descriptors;
    }
    return ContainerUtil.filter(descriptors, input -> input.isAllowedInContextType(ruleType));
  }

  public static Collection<CssPseudoSelectorDescriptor> filterPseudoSelectorDescriptorsByColonPrefix(Collection<? extends CssPseudoSelectorDescriptor> descriptors,
                                                                                                     final int prefixLength) {
    return ContainerUtil.filter(descriptors, input -> input.getColonPrefixLength() == prefixLength);
  }

  public static @NotNull <T extends CssElementDescriptor> Collection<T> sortDescriptors(@NotNull Collection<T> descriptors) {
    if (descriptors.size() <= 1) {
      return descriptors;
    }
    return ContainerUtil.sorted(descriptors, new CssVersionDescriptorComparator());
  }

  private static class CompositeCssElementDescriptorProvider extends CssElementDescriptorProvider {
    private final List<? extends CssElementDescriptorProvider> myProviders;

    CompositeCssElementDescriptorProvider(List<? extends CssElementDescriptorProvider> providers) {
      myProviders = providers;
    }

    @Override
    public boolean isMyContext(@Nullable PsiElement context) {
      for (CssElementDescriptorProvider provider : myProviders) {
        if (provider.isMyContext(context)) {
          return true;
        }
      }
      return false;
    }

    @Override
    public boolean skipCssPropertyCheck(@NotNull CssDeclaration declaration) {
      for (CssElementDescriptorProvider provider : myProviders) {
        if (provider.skipCssPropertyCheck(declaration)) {
          return true;
        }
      }

      return false;
    }

    @Override
    public boolean skipUnknownAtRuleCheck(@NotNull CssAtRule atRule) {
      for (CssElementDescriptorProvider provider : myProviders) {
        if (provider.skipUnknownAtRuleCheck(atRule)) {
          return true;
        }
      }

      return false;
    }

    @Override
    public @NotNull String restoreFullPropertyName(@NotNull String propertyName, @Nullable PsiElement context) {
      for (CssElementDescriptorProvider provider : myProviders) {
        String restored = provider.restoreFullPropertyName(propertyName, context);
        if (!restored.equals(propertyName)) {
          return restored;
        }
      }

      return propertyName;
    }

    @Override
    public @Nullable CssPropertyDescriptor getPropertyDescriptor(@NotNull String propertyName, @Nullable PsiElement context) {
      for (CssElementDescriptorProvider provider : myProviders) {
        final CssPropertyDescriptor descriptor = provider.getPropertyDescriptor(propertyName, context);
        if (descriptor != null) {
          return descriptor;
        }
      }
      return null;
    }

    @Override
    public @NotNull Collection<? extends CssPseudoSelectorDescriptor> findPseudoSelectorDescriptors(@NotNull String name,
                                                                                                    @Nullable PsiElement context) {
      Set<CssPseudoSelectorDescriptor> result = new HashSet<>();
      for (CssElementDescriptorProvider provider : myProviders) {
        result.addAll(provider.findPseudoSelectorDescriptors(name, context));
      }
      return result;
    }

    @Override
    public @NotNull Collection<? extends CssValueDescriptor> getNamedValueDescriptors(@NotNull String name,
                                                                                      @Nullable CssValueDescriptor parent) {
      Set<CssValueDescriptor> result = new HashSet<>();
      for (CssElementDescriptorProvider provider : myProviders) {
        result.addAll(provider.getNamedValueDescriptors(name, parent));
      }
      return result;
    }

    @Override
    public @NotNull Collection<? extends CssPropertyDescriptor> findPropertyDescriptors(@NotNull String propertyName, PsiElement context) {
      Set<CssPropertyDescriptor> result = new HashSet<>();
      for (CssElementDescriptorProvider provider : myProviders) {
        result.addAll(provider.findPropertyDescriptors(propertyName, context));
      }
      return result;
    }

    @Override
    public @NotNull Collection<? extends CssFunctionDescriptor> findFunctionDescriptors(@NotNull String functionName,
                                                                                        @Nullable PsiElement context) {
      Set<CssFunctionDescriptor> result = new HashSet<>();
      for (CssElementDescriptorProvider provider : myProviders) {
        result.addAll(provider.findFunctionDescriptors(functionName, context));
      }
      return result;
    }

    @Override
    public @NotNull Collection<? extends CssMediaFeatureDescriptor> findMediaFeatureDescriptors(@NotNull String mediaFeatureName,
                                                                                                @Nullable PsiElement context) {
      Set<CssMediaFeatureDescriptor> result = new HashSet<>();
      for (CssElementDescriptorProvider provider : myProviders) {
        result.addAll(provider.findMediaFeatureDescriptors(mediaFeatureName, context));
      }
      return result;
    }

    @Override
    public boolean isPossibleSelector(@NotNull String selector, @NotNull PsiElement context) {
      for (CssElementDescriptorProvider provider : myProviders) {
        if (provider.isPossibleSelector(selector, context)) {
          return true;
        }
      }
      return false;
    }

    @Override
    public @NotNull Collection<? extends CssPseudoSelectorDescriptor> getAllPseudoSelectorDescriptors(@Nullable PsiElement context) {
      final Set<CssPseudoSelectorDescriptor> result = new HashSet<>();
      for (CssElementDescriptorProvider provider : myProviders) {
        result.addAll(provider.getAllPseudoSelectorDescriptors(context));
      }
      return result;
    }

    @Override
    public @NotNull Collection<? extends CssPropertyDescriptor> getAllPropertyDescriptors(@Nullable PsiElement context) {
      Set<CssPropertyDescriptor> result = new HashSet<>();
      for (CssElementDescriptorProvider provider : myProviders) {
        result.addAll(provider.getAllPropertyDescriptors(context));
      }
      return result;
    }

    @Override
    public @NotNull Collection<? extends CssMediaFeatureDescriptor> getAllMediaFeatureDescriptors(@Nullable PsiElement context) {
      Set<CssMediaFeatureDescriptor> result = new HashSet<>();
      for (CssElementDescriptorProvider provider : myProviders) {
        result.addAll(provider.getAllMediaFeatureDescriptors(context));
      }
      return result;
    }

    @Override
    public String @NotNull [] getSimpleSelectors(@NotNull PsiElement context) {
      final Set<String> result = new HashSet<>();
      for (CssElementDescriptorProvider provider : myProviders) {
        Collections.addAll(result, provider.getSimpleSelectors(context));
      }
      return ArrayUtilRt.toStringArray(result);
    }

    @Override
    public PsiElement @NotNull [] getDeclarationsForSimpleSelector(@NotNull CssSimpleSelector selector) {
      for (CssElementDescriptorProvider provider : myProviders) {
        final PsiElement[] declarations = provider.getDeclarationsForSimpleSelector(selector);
        if (declarations.length != 0) {
          return declarations;
        }
      }
      return PsiElement.EMPTY_ARRAY;
    }

    @Override
    public boolean providesClassicCss() {
      for (CssElementDescriptorProvider provider : myProviders) {
        if (!provider.providesClassicCss()) {
          return false;
        }
      }
      return true;
    }

    @Override
    public @Nullable PsiElement getDocumentationElementForSelector(@NotNull String selectorName, @Nullable PsiElement context) {
      for (CssElementDescriptorProvider provider : myProviders) {
        final PsiElement element = provider.getDocumentationElementForSelector(selectorName, context);
        if (element != null) {
          return element;
        }
      }
      return null;
    }

    @Override
    public @Nullable String generateDocForSelector(@NotNull String selectorName, @Nullable PsiElement context) {
      for (CssElementDescriptorProvider provider : myProviders) {
        final String doc = provider.generateDocForSelector(selectorName, context);
        if (doc != null) {
          return doc;
        }
      }
      return null;
    }

    @Override
    public @Nullable Color getColorByValue(@NotNull String value) {
      for (CssElementDescriptorProvider provider : myProviders) {
        final Color color = provider.getColorByValue(value);
        if (color != null) {
          return color;
        }
      }
      return null;
    }

    @Override
    public boolean isColorTerm(@NotNull CssTerm term) {
      for (CssElementDescriptorProvider provider : myProviders) {
        if (provider.isColorTerm(term)) {
          return true;
        }
      }
      return false;
    }

    @Override
    public LocalQuickFix @NotNull [] getQuickFixesForUnknownProperty(@NotNull String propertyName,
                                                                     @NotNull PsiElement context,
                                                                     boolean isOnTheFly) {
      final Set<LocalQuickFix> quickFixes = new HashSet<>();
      for (CssElementDescriptorProvider provider : myProviders) {
        Collections.addAll(quickFixes, provider.getQuickFixesForUnknownProperty(propertyName, context, isOnTheFly));
      }
      return quickFixes.toArray(LocalQuickFix.EMPTY_ARRAY);
    }

    @Override
    public LocalQuickFix @NotNull [] getQuickFixesForUnknownSimpleSelector(@NotNull String selectorName,
                                                                           @NotNull PsiElement context,
                                                                           boolean isOnTheFly) {
      final Set<LocalQuickFix> quickFixes = new HashSet<>();
      for (CssElementDescriptorProvider provider : myProviders) {
        Collections.addAll(quickFixes, provider.getQuickFixesForUnknownSimpleSelector(selectorName, context, isOnTheFly));
      }
      return quickFixes.toArray(LocalQuickFix.EMPTY_ARRAY);
    }

    @Override
    public boolean isColorTermsSupported() {
      for (CssElementDescriptorProvider provider : myProviders) {
        if (!provider.isColorTermsSupported()) {
          return false;
        }
      }
      return true;
    }

    @Override
    public CssContextType getCssContextType(@Nullable PsiElement context) {
      for (CssElementDescriptorProvider provider : myProviders) {
        final CssContextType ruleType = provider.getCssContextType(context);
        if (ruleType != CssContextType.ANY) {
          return ruleType;
        }
      }
      return CssContextType.ANY;
    }

    @Override
    public @NotNull PsiReference getStyleReference(PsiElement element, int start, int end, boolean caseSensitive) {
      for (CssElementDescriptorProvider provider : myProviders) {
        PsiReference reference = provider.getStyleReference(element, start, end, caseSensitive);
        if (!(reference instanceof CssStyleReferenceStub)) {
          return reference;
        }
      }
      return new CssStyleReferenceStub(element, TextRange.create(start, end));
    }

    @Override
    public @NotNull CssValueValidator getValueValidator() {
      for (CssElementDescriptorProvider provider : myProviders) {
        CssValueValidator validator = provider.getValueValidator();
        if (!(validator instanceof CssValueValidatorStub)) {
          return validator;
        }
      }
      return super.getValueValidator();
    }
  }
}
