package com.intellij.psi.css.impl.util;

import com.google.gson.Gson;
import com.intellij.openapi.util.Comparing;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.css.descriptor.BrowserVersion;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class CssCompatibilityData {
  private static final Map<String, Map> PROPERTIES_CACHE = new HashMap<>();
  private static final Map<String, Map> PSEUDO_SELECTOR_CACHE = new HashMap<>();

  @Nullable
  public static Map getPropertyData(@NotNull String propertyName) {
    return getCssCompatData(PROPERTIES_CACHE, "properties", propertyName);
  }

  @Nullable
  public static Map getPseudoSelectorData(@NotNull String selectorName) {
    return getCssCompatData(PSEUDO_SELECTOR_CACHE, "selectors", selectorName);
  }

  @Nullable
  private static Map getCssCompatData(Map<String, Map> cache, String cssCompatDataType, @NotNull String name) {
    if (cache.containsKey(name)) {
      return cache.get(name);
    }

    URL resource = CssCompatibilityData.class.getResource("/cssCompatData/" + cssCompatDataType + "/" + name + ".json");
    if (resource == null) {
      cache.put(name, null);
      return null;
    }

    try {
      Map json = new Gson().fromJson(new InputStreamReader(resource.openStream(), StandardCharsets.UTF_8), Map.class);
      Map propertiesHolder = (Map)json.get("css");
      Map properties = (Map)propertiesHolder.get(cssCompatDataType);
      Map data = (Map)properties.get(name);
      Map validData = findBestData(data);
      cache.put(name, validData);
      return validData;
    }
    catch (IOException e) {
      cache.put(name, null);
      return null;
    }
  }

  @NotNull
  private static Map findBestData(@NotNull Map data) {
    // Properties like 'align-content' have more than one data map. We'll choose the one with the best compat data (earlier browser versions)

    if (data.get("__compat") != null) {
      return data;
    }

    Map bestData = null;
    BrowserVersion[] minimalVersions = BrowserVersion.EMPTY_ARRAY;
    for (Object subData : data.values()) {
      if (bestData == null) {
        bestData = (Map)subData;
        minimalVersions = getBrowsersData((Map)subData);
      }
      else {
        BrowserVersion[] candidate = getBrowsersData((Map)subData);
        if (isFirstBetter(candidate, minimalVersions)) {
          bestData = (Map)subData;
          minimalVersions = candidate;
        }
      }
    }

    return bestData != null ? bestData : data;
  }

  private static boolean isFirstBetter(BrowserVersion[] candidate1, BrowserVersion[] candidate2) {
    int score = 0;
    for (BrowserVersion.Browser browser : BrowserVersion.Browser.values()) {
      BrowserVersion version1 = ContainerUtil.find(candidate1, version -> version.getBrowser() == browser);
      BrowserVersion version2 = ContainerUtil.find(candidate2, version -> version.getBrowser() == browser);
      score += Comparing.compare(version1, version2, (o1, o2) -> {
        String v1 = o1.getVersion();
        String v2 = o2.getVersion();
        if (v1.equals(v2)) return 0;
        if (v1.equals("true")) return 1;
        if (v2.equals("true")) return -1;
        // arguments order (first v2, second v1) is not a typo here!
        return Integer.signum(StringUtil.compareVersionNumbers(v2, v1));
      });
    }

    return score > 0;
  }

  public static BrowserVersion @NotNull [] getBrowsersDataForProperty(String propertyName) {
    return getBrowsersData(getPropertyData(propertyName));
  }

  public static BrowserVersion[] getBrowsersDataForPseudoSelector(String selectorName) {
    return getBrowsersData(getPseudoSelectorData(selectorName));
  }

  private static BrowserVersion @NotNull [] getBrowsersData(Map mdnCompatData) {
    if (mdnCompatData == null) return BrowserVersion.EMPTY_ARRAY;

    Map compat = (Map)mdnCompatData.get("__compat");
    if (compat == null) return BrowserVersion.EMPTY_ARRAY;

    Map support = (Map)compat.get("support");
    if (support == null) return BrowserVersion.EMPTY_ARRAY;

    List<BrowserVersion> result = new ArrayList<>();
    for (Object key : support.keySet()) {
      String browserId = key.toString();
      BrowserVersion.Browser browser = getBrowserByMdnId(browserId);
      if (browser == null) continue;

      Object browserInfos = support.get(key);
      Map browserInfo = (Map)(browserInfos instanceof ArrayList ? ((ArrayList)browserInfos).get(0) : browserInfos);
      Object versionAdded = browserInfo.get("version_added");
      String version = versionAdded != null ? versionAdded.toString() : "";
      if (!StringUtil.isEmpty(version) && !"false".equals(version)) {
        result.add(new BrowserVersion(browser, version));
      }
    }

    return result.toArray(BrowserVersion.EMPTY_ARRAY);
  }

  @Nullable
  private static BrowserVersion.Browser getBrowserByMdnId(String mdnBrowserId) {
    switch (mdnBrowserId) {
      case "chrome":
        return BrowserVersion.Browser.CHROME;
      case "firefox":
        return BrowserVersion.Browser.FIREFOX;
      case "ie":
        return BrowserVersion.Browser.IE;
      case "opera":
        return BrowserVersion.Browser.OPERA;
      case "safari":
        return BrowserVersion.Browser.SAFARI;
      case "edge":
        return BrowserVersion.Browser.EDGE;
      default:
        return null;
    }
  }
}
