package com.intellij.psi.css.descriptor;

import com.intellij.psi.PsiElement;
import com.intellij.psi.css.impl.util.table.CssDescriptorsUtil;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;
import java.util.Collection;
import java.util.Collections;
import java.util.Set;
import java.util.TreeSet;

public final class CssCompositeElementDescriptor implements CssElementDescriptor {
  @NotNull private final Collection<? extends CssElementDescriptor> myDescriptors;
  @NotNull private final CssElementDescriptor myDescriptorFromLatestSpec;

  @Nullable
  public static CssElementDescriptor create(@NotNull Collection<? extends CssElementDescriptor> descriptors) {
    if (descriptors.isEmpty()) {
      return null;
    }
    if (descriptors.size() == 1) {
      return ContainerUtil.getFirstItem(descriptors);
    }
    return new CssCompositeElementDescriptor(descriptors);
  }

  private CssCompositeElementDescriptor(@NotNull Collection<? extends CssElementDescriptor> descriptors) {
    assert descriptors.size() > 0;
    myDescriptors = descriptors;
    CssElementDescriptor descriptorFromLatestSpec = CssDescriptorsUtil.getDescriptorFromLatestSpec(descriptors);
    assert descriptorFromLatestSpec != null;
    myDescriptorFromLatestSpec = descriptorFromLatestSpec;
  }

  @NotNull
  @Override
  public CssVersion getCssVersion() {
    CssVersion result = CssVersion.UNKNOWN;
    for (CssElementDescriptor descriptor : myDescriptors) {
      CssVersion version = descriptor.getCssVersion();
      if (result == CssVersion.UNKNOWN || result.value() > version.value()) {
        result = version;
      }
    }
    return result;
  }

  @Override
  public BrowserVersion @NotNull [] getBrowsers() {
    Set<BrowserVersion> browsers = new TreeSet<>(BrowserVersion.COMPARATOR);
    for (CssElementDescriptor descriptor : myDescriptors) {
      final BrowserVersion[] descriptorBrowsers = descriptor.getBrowsers();
      if (descriptorBrowsers.length == 0) {
        return BrowserVersion.EMPTY_ARRAY;
      }
      Collections.addAll(browsers, descriptorBrowsers);
    }
    return browsers.toArray(BrowserVersion.EMPTY_ARRAY);
  }

  @Override
  public boolean isAllowedInContextType(@NotNull CssContextType contextType) {
    for (CssElementDescriptor descriptor : myDescriptors) {
      if (descriptor.isAllowedInContextType(contextType)) {
        return true;
      }
    }
    return false;
  }

  @NotNull
  @Override
  public String getId() {
    return myDescriptorFromLatestSpec.getId();
  }

  @NotNull
  @Override
  public String getPresentableName() {
    return myDescriptorFromLatestSpec.getPresentableName();
  }

  @NotNull
  @Override
  public String getDescription() {
    return myDescriptorFromLatestSpec.getPresentableName();
  }

  @Nullable
  @Override
  public String getDocumentationString(@Nullable PsiElement context) {
    return myDescriptorFromLatestSpec.getDocumentationString(context);
  }

  @NotNull
  @Override
  public String getElementTypeName() {
    return myDescriptorFromLatestSpec.getElementTypeName();
  }

  @Nullable
  @Override
  public String getSpecificationUrl() {
    return myDescriptorFromLatestSpec.getSpecificationUrl();
  }

  @Override
  public CssContextType @NotNull [] getAllowedContextTypes() {
    Set<CssContextType> contextTypes = new TreeSet<>(CssContextType.COMPARATOR);
    for (CssElementDescriptor descriptor : myDescriptors) {
      Collections.addAll(contextTypes, descriptor.getAllowedContextTypes());
    }
    return contextTypes.toArray(CssContextType.EMPTY_ARRAY);
  }

  @Nullable
  @Override
  public Icon getIcon() {
    return myDescriptorFromLatestSpec.getIcon();
  }

  @NotNull
  public CssElementDescriptor getDescriptorFromLatestSpec() {
    return myDescriptorFromLatestSpec;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (!(o instanceof CssCompositeElementDescriptor)) return false;

    CssCompositeElementDescriptor that = (CssCompositeElementDescriptor)o;

    if (!myDescriptors.equals(that.myDescriptors)) return false;

    return true;
  }

  @Override
  public int hashCode() {
    return myDescriptors.hashCode();
  }
}
