package com.intellij.psi.css.descriptor;

import com.intellij.icons.AllIcons;
import org.jetbrains.annotations.NotNull;

import javax.swing.*;
import java.util.Comparator;

public class BrowserVersion {
  public static final BrowserVersion[] EMPTY_ARRAY = new BrowserVersion[0];
  public static final Comparator<BrowserVersion> COMPARATOR = Comparator.comparing(o -> o.getBrowser().getPresentableName());
  
  @NotNull private final Browser myBrowser;
  @NotNull private final String myVersion;

  public BrowserVersion(@NotNull Browser browser, @NotNull String version) {
    myBrowser = browser;
    myVersion = version;
  }

  @NotNull
  @Override
  public String toString() {
    return myBrowser.getPresentableName() + ' ' + myVersion;
  }

  @NotNull
  public String getVersion() {
    return myVersion;
  }

  @NotNull
  public Browser getBrowser() {
    return myBrowser;
  }

  @NotNull
  public static BrowserVersion fromString(@NotNull String string) {
    if (!string.isEmpty()) {
      switch (string.charAt(0)) {
        case 'C':
          return new BrowserVersion(Browser.CHROME, string.substring(1));
        case 'F': //FF expected
          return new BrowserVersion(Browser.FIREFOX, string.substring(2));
        case 'I': //IE expected
          return new BrowserVersion(Browser.IE, string.substring(2));
        case 'O':
          return new BrowserVersion(Browser.OPERA, string.substring(1));
        case 'S':
          return new BrowserVersion(Browser.SAFARI, string.substring(1));
        case 'E':
          return new BrowserVersion(Browser.EDGE, string.substring(1));
      }
    }
    throw new IllegalArgumentException("Can't create browser version for " + string);
  }

  public enum Browser {
    CHROME("Chrome", AllIcons.Xml.Browsers.Chrome),
    FIREFOX("Firefox", AllIcons.Xml.Browsers.Firefox),
    SAFARI("Safari", AllIcons.Xml.Browsers.Safari),
    OPERA("Opera", AllIcons.Xml.Browsers.Opera),
    IE("IE", AllIcons.Xml.Browsers.Explorer),
    EDGE("Edge", AllIcons.Xml.Browsers.Edge);

    @NotNull private final Icon myIcon;
    @NotNull private final String myPresentableName;

    Browser(@NotNull String presentableName, @NotNull Icon icon) {
      myIcon = icon;
      myPresentableName = presentableName;
    }

    @NotNull
    public Icon getIcon() {
      return myIcon;
    }

    @NotNull
    public String getPresentableName() {
      return myPresentableName;
    }
  }

}
