package com.intellij.psi.css;

import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiElement;
import com.intellij.psi.SmartPsiElementPointer;
import com.intellij.psi.css.resolve.CssResolver;
import com.intellij.psi.xml.XmlTag;
import com.intellij.util.Processor;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Map;

public interface CssSelector extends CssElement {
  Key<SmartPsiElementPointer<CssSelector>> AMPERSAND_SELECTOR_ORIGIN_KEY = Key.create("AMPERSAND_SELECTOR_ORIGIN_KEY");

  enum Combinator {
    DESCENDANT(' '),
    CHILD('>'),
    SIBLING('+'),
    FOLLOWING_SIBLING('~'),
    COLUMN('|', "||");

    private final char myIdentifier;
    private final String myPresentation;

    Combinator(char identifier, String presentation) {
      myIdentifier = identifier;
      myPresentation = presentation;
    }

    Combinator(char identifier) {
      this(identifier, null);
    }

    public char getIdentifier() {
      return myIdentifier;
    }

    public String getPresentation() {
      return StringUtil.notNullize(myPresentation, String.valueOf(myIdentifier));
    }

    public static Combinator fromIdentifier(char c) {
      for (Combinator combinator : values()) {
        if (combinator.myIdentifier == c) {
          return combinator;
        }
      }
      throw new IllegalArgumentException("Illegal combinator identifier: " + c);
    }
  }

  CssSelector[] EMPTY_ARRAY = new CssSelector[0];

  @NotNull
  String getPresentableText();

  /**
   * @return CssSimpleSelectors, XhtmlTokens (CSS_GT ('>'), CSS_PLUS ('+'))
   * @deprecated breaks stub tree, use {@link this#getCombinators()} and {@link this#getSimpleSelectors()} instead
   */
  @Deprecated
  PsiElement[] getElements();

  Combinator @NotNull [] getCombinators();

  CssSimpleSelector @NotNull [] getSimpleSelectors();

  @Nullable SelectorSpecificity getSpecificity();

  boolean processAmpersandEvaluatedSelectors(@NotNull Processor<? super CssSelector> processor);

  boolean isAmpersandSelector();

  boolean isMatch(@NotNull XmlTag tag, @NotNull CssResolver resolver, @Nullable Map<XmlTag, CssSimpleSelector> mapping);

  /**
   * Returns <code>true</code> if all following conditions are met: <ul>
   * <li>the passed <code>tag</code> doesn't match <code>this</code> selector,</li>
   * <li>the passed <code>tag</code> has CSS class(es) specified via <code>class</code> attribute,</li>
   * <li><code>this</code> selector is hierarchical,</li>
   * <li><code>this</code> selector contains matching class in chain (but not the last one),</li>
   * <li>any of the <code>tag</code>'s subtags matches <code>this</code> selector.</li>
   * </ul>
   * Example: this method will return <code>true</code> for CSS selector <br/>
   * <code>.fooClass li {}</code><br/>
   * and this <code>&lt;ul&gt;</code> tag:<br/>
   * <code>&lt;ul class='fooClass'&gt; &lt;li&gt;&lt;/li&gt; &lt;/ul&gt;</code>
   */
  boolean isHierarchicalSelectorThatMatchesSubtag(@NotNull XmlTag tag, @NotNull CssResolver resolver);

  @Nullable
  CssRuleset getRuleset();

  int getLineNumber();
}
