// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.model.autoconfigure;

import com.intellij.openapi.util.Comparing;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiClass;
import com.intellij.spring.model.jam.SpringOrder;
import com.intellij.util.ObjectUtils;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;

/**
 * @author Konstantin Aleev
 */
public class SpringOrderClassSorter {

  public static final Comparator<PsiClass> CLASS_NAME_COMPARATOR =
    (o1, o2) -> StringUtil.compare(o1.getQualifiedName(), o2.getQualifiedName(), false);

  private static final Comparator<PsiClass> ORDER_COMPARATOR = new Comparator<>() {

    @Override
    public int compare(PsiClass o1, PsiClass o2) {
      final int order1 = getOrderValue(o1);
      final int order2 = getOrderValue(o2);
      return Comparing.compare(order1, order2);
    }

    private int getOrderValue(PsiClass psiClass) {
      final SpringOrder order = SpringOrder.CLASS_META.getJamElement(psiClass);
      if (order == null) return SpringOrder.LOWEST_PRECEDENCE;
      return ObjectUtils.notNull(order.getValue(), SpringOrder.LOWEST_PRECEDENCE);
    }
  };

  private final List<? extends PsiClass> allConfigs;

  public SpringOrderClassSorter(List<? extends PsiClass> allConfigs) {
    this.allConfigs = allConfigs;
  }

  public List<PsiClass> getSortedConfigs() {
    // sort by FQN
    List<PsiClass> configs = new ArrayList<>(allConfigs);
    configs.sort(CLASS_NAME_COMPARATOR);

    // sort by @Order
    configs.sort(ORDER_COMPARATOR);

    return configs;
  }
}
