// Copyright 2000-2017 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.model;

import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.roots.ModuleRootManager;
import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.NlsContexts;
import com.intellij.openapi.util.NlsSafe;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiClass;
import com.intellij.spring.boot.application.SpringBootApplicationService;
import com.intellij.spring.facet.SpringFacet;
import com.intellij.spring.facet.SpringFileSet;
import com.intellij.spring.facet.beans.CustomSetting;
import org.jetbrains.annotations.Nls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;
import java.util.Collections;
import java.util.List;

/**
 * Provides file names (default ({@code application|bootstrap}) and/or custom named)
 * used by {@link SpringBootModelConfigFileContributor} to collect config files.
 *
 * @author Konstantin Aleev
 */
public abstract class SpringBootModelConfigFileNameContributor {
  public static final ExtensionPointName<SpringBootModelConfigFileNameContributor> EP_NAME =
    ExtensionPointName.create("com.intellij.spring.boot.modelConfigFileNameContributor");

  private final CustomSettingDescriptor myCustomNameSettingDescriptor;
  private final CustomSettingDescriptor myCustomFilesSettingDescriptor;
  private final CustomizationPresentation myCustomizationPresentation;
  private final Icon myFileIcon;

  protected SpringBootModelConfigFileNameContributor(@NotNull CustomSettingDescriptor customNameSettingDescriptor,
                                                     @NotNull CustomSettingDescriptor customFilesSettingDescriptor,
                                                     @NotNull CustomizationPresentation presentation,
                                                     @NotNull Icon fileIcon) {
    myCustomNameSettingDescriptor = customNameSettingDescriptor;
    myCustomFilesSettingDescriptor = customFilesSettingDescriptor;
    myCustomizationPresentation = presentation;
    myFileIcon = fileIcon;
  }

  @NotNull
  public final CustomSettingDescriptor getCustomNameSettingDescriptor() {
    return myCustomNameSettingDescriptor;
  }

  @NotNull
  public final CustomSettingDescriptor getCustomFilesSettingDescriptor() {
    return myCustomFilesSettingDescriptor;
  }

  @NotNull
  public CustomizationPresentation getCustomizationPresentation() {
    return myCustomizationPresentation;
  }

  @NotNull
  public Icon getFileIcon() {
    return myFileIcon;
  }

  public abstract boolean accept(SpringFileSet fileSet);

  public abstract boolean accept(Module module);

  public @NotNull String getSpringConfigName(@NotNull Module module) {
    SpringFacet springFacet = getRelevantFacet(module);
    if (springFacet == null) { // adding new facet -> edit dialog
      return getCustomNameSettingDescriptor().defaultValue;
    }

    CustomSetting.STRING setting = springFacet.findSetting(getCustomNameSettingDescriptor().key);
    assert setting != null;

    return StringUtil.notNullize(setting.getStringValue(), setting.getDefaultValue());
  }

  public @NotNull List<String> getCustomConfigFileUrls(@NotNull Module module) {
    final SpringFacet springFacet = getRelevantFacet(module);
    if (springFacet == null) { // adding new facet -> edit dialog
      return Collections.emptyList();
    }

    CustomSetting.STRING setting = springFacet.findSetting(getCustomFilesSettingDescriptor().key);
    assert setting != null;
    String value = setting.getStringValue();
    if (value == null) {
      return Collections.emptyList();
    }

    return StringUtil.split(value, ";");
  }

  @Nullable
  private static SpringFacet getRelevantFacet(Module module) {
    SpringFacet facet = findSpringBootApplicationFacet(module);
    if (facet != null) return facet;

    if (module.isDisposed()) return null;

    // Gradle: test-only module has dependency to main module with SB main class + possible custom configuration
    for (Module dependentModule : ModuleRootManager.getInstance(module).getDependencies()) {
      final SpringFacet dependentFacet = findSpringBootApplicationFacet(dependentModule);
      if (dependentFacet != null) {
        return dependentFacet;
      }
    }

    return null;
  }

  @Nullable
  private static SpringFacet findSpringBootApplicationFacet(Module module) {
    SpringFacet springFacet = SpringFacet.getInstance(module);
    if (springFacet == null) return null;

    final List<PsiClass> applications = SpringBootApplicationService.getInstance().getSpringApplications(module);
    return applications.isEmpty() ? null : springFacet;
  }

  public static class CustomSettingDescriptor {
    public final Key<CustomSetting.STRING> key;
    public final @Nls String description;
    public final String defaultValue;

    public CustomSettingDescriptor(Key<CustomSetting.STRING> key, @Nls String description, String defaultValue) {
      this.key = key;
      this.description = description;
      this.defaultValue = defaultValue;
    }

    public CustomSetting.STRING createCustomSetting() {
      return new CustomSetting.STRING(key, description, defaultValue);
    }
  }

  public static class CustomizationPresentation {
    public final @NlsSafe String configFileKey;
    public final @NlsContexts.BorderTitle String customizationPanelTitle;

    public CustomizationPresentation(@NlsSafe String configFileKey,
                                     @NlsContexts.BorderTitle String customizationPanelTitle) {
      this.configFileKey = configFileKey;
      this.customizationPanelTitle = customizationPanelTitle;
    }
  }
}
