// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.library;

import com.intellij.compiler.CompilerConfiguration;
import com.intellij.jam.JavaLibraryUtils;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.roots.ProjectRootManager;
import com.intellij.openapi.roots.libraries.JarVersionDetectionUtil;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.patterns.PatternCondition;
import com.intellij.psi.JavaPsiFacade;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElement;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.psi.util.CachedValuesManager;
import com.intellij.spring.boot.SpringBootClassesConstants;
import com.intellij.spring.model.utils.SpringCommonUtils;
import com.intellij.util.ArrayUtil;
import com.intellij.util.ProcessingContext;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.jps.model.java.compiler.AnnotationProcessingConfiguration;

import java.io.File;
import java.util.Set;

import static com.intellij.psi.util.CachedValueProvider.Result;

public final class SpringBootLibraryUtil {

  public static final PatternCondition<PsiElement> SB_1_3_OR_HIGHER =
    createVersionPatternCondition(SpringBootVersion.VERSION_1_3_0);

  public static final PatternCondition<PsiElement> SB_1_5_OR_HIGHER =
    createVersionPatternCondition(SpringBootVersion.VERSION_1_5_0);

  public static final PatternCondition<PsiElement> SB_2_0_OR_HIGHER =
    createVersionPatternCondition(SpringBootVersion.VERSION_2_0_0);

  private static final String CONFIGURATION_PROCESSOR_ARTIFACT_NAME = "spring-boot-configuration-processor";

  /**
   * Creates condition with minimum required SB version.
   *
   * @param minimumVersion Minimum required SB version.
   * @return PatternCondition.
   */
  public static PatternCondition<PsiElement> createVersionPatternCondition(SpringBootVersion minimumVersion) {
    return new PatternCondition<>("SB" + minimumVersion.name() + "OrHigher") {
      @Override
      public boolean accepts(@NotNull PsiElement element, ProcessingContext context) {
        final Module module = ModuleUtilCore.findModuleForPsiElement(element);
        return isAtLeastVersion(module, minimumVersion);
      }
    };
  }

  // must be listed in ascending order
  public enum SpringBootVersion {
    ANY(SpringBootClassesConstants.SPRING_APPLICATION),
    VERSION_1_2_0("org.springframework.boot.jta.XADataSourceWrapper"),
    VERSION_1_3_0("org.springframework.boot.context.event.ApplicationReadyEvent"),
    VERSION_1_4_0("org.springframework.boot.diagnostics.FailureAnalysis"),
    VERSION_1_5_0("org.springframework.boot.autoconfigure.AbstractDatabaseInitializer"),
    VERSION_2_0_0("org.springframework.boot.WebApplicationType"),

    VERSION_2_1_0("org.springframework.boot.context.event.ApplicationContextInitializedEvent"),
    VERSION_2_2_0("org.springframework.boot.context.properties.ConfigurationPropertiesScan"),
    VERSION_2_4_0("org.springframework.boot.context.config.ConfigData");

    private final String myDetectionClassFqn;

    SpringBootVersion(String detectionClassFqn) {
      myDetectionClassFqn = detectionClassFqn;
    }

    public boolean isAtLeast(SpringBootVersion reference) {
      if (reference == ANY) {
        return true;
      }
      return compareTo(reference) >= 0;
    }

    public String getDetectionClassFqn() {
      return myDetectionClassFqn;
    }
  }

  public static boolean hasConfigurationMetadataAnnotationProcessor(@NotNull Module module) {
    AnnotationProcessingConfiguration configuration =
      CompilerConfiguration.getInstance(module.getProject()).getAnnotationProcessingConfiguration(module);
    if (!configuration.isEnabled()) {
      return false;
    }

    Set<String> processors = configuration.getProcessors();
    if (!processors.isEmpty() && !processors.contains(SpringBootClassesConstants.CONFIGURATION_METADATA_ANNOTATION_PROCESSOR)) {
      return false;
    }

    if (configuration.isObtainProcessorsFromClasspath()) {
      PsiClass processor = SpringCommonUtils.findLibraryClass(module,
                                                              SpringBootClassesConstants.CONFIGURATION_METADATA_ANNOTATION_PROCESSOR);
      return processor != null;
    }

    Iterable<String> segments = StringUtil.tokenize(configuration.getProcessorPath(), File.pathSeparator);
    for (String segment : segments) {
      if (segment.endsWith(".jar")) {
        int fileNameIndex = Integer.max(segment.lastIndexOf('/'), segment.lastIndexOf('\\'));
        if (fileNameIndex >= 0 && fileNameIndex < segment.length() - 1) {
          segment = segment.substring(fileNameIndex + 1);
        }
        if (segment.contains(CONFIGURATION_PROCESSOR_ARTIFACT_NAME)) {
          return true;
        }
      }
    }

    return false;
  }

  public static boolean hasSpringBootLibrary(Project project) {
    return JavaLibraryUtils.hasLibraryClass(project, SpringBootVersion.ANY.getDetectionClassFqn());
  }

  public static boolean hasSpringBootLibrary(@Nullable final Module module) {
    return isAtLeastVersion(module, SpringBootVersion.ANY);
  }

  public static boolean isBelowVersion(@Nullable final Module module,
                                       final SpringBootVersion version) {
    return !isAtLeastVersion(module, version);
  }

  public static boolean isAtLeastVersion(@Nullable final Module module,
                                         final SpringBootVersion version) {
    if (module == null || module.isDisposed()) return false;

    if (!hasSpringBootLibrary(module.getProject())) return false;

    final SpringBootVersion cached = getSpringBootVersion(module);
    return cached != null && cached.isAtLeast(version);
  }

  @Nullable
  public static String getVersionFromJar(@NotNull Module module) {
    return JarVersionDetectionUtil.detectJarVersion(SpringBootVersion.ANY.getDetectionClassFqn(), module);
  }

  public static boolean hasDevtools(@Nullable final Module module) {
    return SpringCommonUtils.findLibraryClass(module, SpringBootClassesConstants.REMOTE_SPRING_APPLICATION) != null;
  }

  public static boolean hasActuators(@Nullable final Module module) {
    if (isAtLeastVersion(module, SpringBootVersion.VERSION_2_0_0)) {
      return SpringCommonUtils.findLibraryClass(module, "org.springframework.boot.actuate.endpoint.annotation.Endpoint") != null;
    }
    return SpringCommonUtils.findLibraryClass(module, "org.springframework.boot.actuate.endpoint.Endpoint") != null;
  }

  public static boolean hasRequestMappings(@Nullable final Module module) {
    return SpringCommonUtils.findLibraryClass(module, "org.springframework.web.servlet.handler.AbstractHandlerMethodMapping") != null ||
           SpringCommonUtils.findLibraryClass(module, "org.springframework.web.reactive.result.method.AbstractHandlerMethodMapping") != null;
  }

  @Nullable
  public static SpringBootVersion getSpringBootVersion(@NotNull final Module module) {
    final Project project = module.getProject();

    return CachedValuesManager.getManager(project).getCachedValue(module, () -> {
      final GlobalSearchScope scope = GlobalSearchScope.moduleRuntimeScope(module, false);
      final JavaPsiFacade javaPsiFacade = JavaPsiFacade.getInstance(project);

      SpringBootVersion detected = null;
      for (SpringBootVersion version : ArrayUtil.reverseArray(SpringBootVersion.values())) {
        final PsiClass psiClass = javaPsiFacade.findClass(version.getDetectionClassFqn(), scope);
        if (psiClass != null) {
          detected = version;
          break;
        }
      }

      return Result.create(detected, ProjectRootManager.getInstance(project));
    });
  }
}
