// Copyright 2000-2017 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.application.metadata.additional;

import com.intellij.json.psi.JsonFile;
import com.intellij.openapi.application.ApplicationManager;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.roots.ModuleRootManager;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.psi.PsiFile;
import com.intellij.psi.PsiManager;
import com.intellij.spring.boot.SpringBootConfigFileConstants;
import com.intellij.util.Processor;
import org.jetbrains.annotations.TestOnly;
import org.jetbrains.jps.model.java.JavaResourceRootType;
import org.jetbrains.jps.model.java.JavaSourceRootType;
import org.jetbrains.jps.model.module.JpsModuleSourceRootType;

import java.util.List;

public class SpringBootAdditionalConfigUtils {

  private final Module myModule;
  private final List<VirtualFile> myResourceRoots;

  public SpringBootAdditionalConfigUtils(Module module) {
    myModule = module;

    final boolean testMode = ApplicationManager.getApplication().isUnitTestMode();
    final JpsModuleSourceRootType<?> sourceRootType = testMode ? JavaSourceRootType.SOURCE : JavaResourceRootType.RESOURCE;
    myResourceRoots = ModuleRootManager.getInstance(module).getSourceRoots(sourceRootType);
  }

  boolean hasResourceRoots() {
    return !myResourceRoots.isEmpty();
  }

  List<VirtualFile> getResourceRoots() {
    return myResourceRoots;
  }

  public boolean processAllAdditionalMetadataFiles(Processor<? super PsiFile> processor) {
    for (VirtualFile root : myResourceRoots) {
      final VirtualFile vf = root.findFileByRelativePath("META-INF/" +
                                                         SpringBootConfigFileConstants.ADDITIONAL_SPRING_CONFIGURATION_METADATA_JSON);
      if (vf == null) continue;
      PsiFile psiFile = PsiManager.getInstance(myModule.getProject()).findFile(vf);
      if (!processor.process(psiFile)) return false;
    }
    return true;
  }

  public boolean processAdditionalMetadataFiles(Processor<? super JsonFile> processor) {
    return processAllAdditionalMetadataFiles(psiFile -> {
      if (psiFile instanceof JsonFile) {
        if (!processor.process((JsonFile)psiFile)) return false;
      }
      return true;
    });
  }

  private static boolean OVERRIDE = false;

  @TestOnly
  public static void setOverrideDetection(boolean value) {
    OVERRIDE = value;
  }

  public static boolean isAdditionalMetadataFile(PsiFile psiFile) {
    if (OVERRIDE) return true;
    
    return psiFile instanceof JsonFile &&
           psiFile.getName().equals(SpringBootConfigFileConstants.ADDITIONAL_SPRING_CONFIGURATION_METADATA_JSON);
  }
}
