// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.application.config;

import com.intellij.openapi.util.Couple;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.util.SmartList;

import java.util.ArrayList;
import java.util.List;

/**
 * Provide access to configured replacement tokens.
 */
public class SpringBootReplacementTokenStorage {

  private final List<String> myTokens = new SmartList<>();

  public void deserialize(String input) {
    myTokens.clear();
    myTokens.addAll(StringUtil.split(input, " "));
  }

  public String serialize() {
    return StringUtil.join(myTokens, " ");
  }

  public List<String> getTokens() {
    return myTokens;
  }

  /**
   * Returns list of valid tokens as {@code prefix|suffix}.
   *
   * @return Valid tokens.
   */
  public List<Couple<String>> getReplacementTokens() {
    List<Couple<String>> tokenList = new ArrayList<>(myTokens.size());
    for (String value : myTokens) {
      if (StringUtil.isEmptyOrSpaces(value)) continue;

      if (StringUtil.containsChar(value, '*')) {
        String prefix = StringUtil.substringBefore(value, "*");
        String suffix = StringUtil.substringAfter(value, "*");
        if (StringUtil.isEmptyOrSpaces(prefix) || StringUtil.isEmptyOrSpaces(suffix)) continue;
        tokenList.add(Couple.of(prefix, suffix));
      }
      else {
        tokenList.add(Couple.of(value, value));
      }
    }
    return tokenList;
  }
}
