// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.application.config;

import com.intellij.codeInsight.lookup.LookupElement;
import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.openapi.util.TextRange;
import com.intellij.psi.*;
import com.intellij.util.ArrayUtil;
import com.intellij.util.SmartList;
import org.jetbrains.annotations.NotNull;

import java.util.List;

/**
 * Resolves replacement tokens ("@my.property@") in configuration files to external definitions, e.g. build files.
 */
public abstract class SpringBootReplacementTokenResolver {

  public static final ExtensionPointName<SpringBootReplacementTokenResolver> EP_NAME =
    ExtensionPointName.create("com.intellij.spring.boot.replacementTokenResolver");

  @NotNull
  public abstract List<PsiElement> resolve(PsiReference reference);

  @NotNull
  public abstract List<LookupElement> getVariants(PsiReference reference);

  public static PsiReference createReference(PsiElement element, TextRange range) {
    return new ReplacementTokenReference(element, range);
  }


  public static final class ReplacementTokenReference extends PsiReferenceBase.Poly<PsiElement> {

    private ReplacementTokenReference(PsiElement element, TextRange rangeInElement) {
      super(element, rangeInElement, true);
    }

    @Override
    public ResolveResult @NotNull [] multiResolve(boolean incompleteCode) {
      List<PsiElement> allResults = new SmartList<>();
      for (SpringBootReplacementTokenResolver replacementTokenResolver : EP_NAME.getExtensions()) {
        allResults.addAll(replacementTokenResolver.resolve(this));
      }
      return PsiElementResolveResult.createResults(allResults);
    }

    @Override
    public Object @NotNull [] getVariants() {
      List<LookupElement> variants = new SmartList<>();
      for (SpringBootReplacementTokenResolver replacementTokenResolver : EP_NAME.getExtensions()) {
        variants.addAll(replacementTokenResolver.getVariants(this));
      }
      return ArrayUtil.toObjectArray(variants);
    }
  }
}
