// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.application.config;

import com.intellij.microservices.config.MetaConfigKey;
import com.intellij.openapi.application.ApplicationManager;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.Condition;
import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.TextRange;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiReference;
import com.intellij.util.ProcessingContext;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

import static com.intellij.microservices.config.MetaConfigKey.AccessType;
import static com.intellij.microservices.config.MetaConfigKey.ItemHint;

/**
 * Gets references provided by SB 1.3 metadata and/or manual metadata.
 */
public abstract class SpringBootHintReferencesProvider {

  /**
   * Map/Index/List key with POJO property path.
   */
  public static final Condition<MetaConfigKey> MAP_OR_INDEXED_WITHOUT_KEY_HINTS_CONDITION = configKey ->
    configKey.getKeyItemHint() == ItemHint.NONE &&
    configKey.isAccessType(AccessType.ENUM_MAP, AccessType.MAP, AccessType.INDEXED);

  public static SpringBootHintReferencesProvider getInstance() {
    return ApplicationManager.getApplication().getService(SpringBootHintReferencesProvider.class);
  }

  /**
   * Available in ProcessingContext.
   */
  public static final Key<MetaConfigKey> HINT_REFERENCES_CONFIG_KEY = Key.create("HINT_REFERENCES_CONFIG_KEY");

  /**
   * Config key text in properties format.
   * Callers of this interface must provide it via ProcessingContext.
   */
  public static final Key<String> HINT_REFERENCES_CONFIG_KEY_TEXT = Key.create("HINT_REFERENCES_CONFIG_KEY_TEXT");

  /**
   * Gets additional key references (via configured hints).
   *
   * @param module        Current module.
   * @param configKey     Current config key.
   * @param keyPsiElement Current key element.
   * @param context       Context, {@link #HINT_REFERENCES_CONFIG_KEY} is made available.
   * @return References.
   */
  public abstract PsiReference @NotNull [] getKeyReferences(Module module,
                                                            MetaConfigKey configKey,
                                                            PsiElement keyPsiElement,
                                                            ProcessingContext context);

  /**
   * Gets value references (hints, by type, key POJO path, explicit/fallback config).
   *
   * @param module          Current module.
   * @param configKey       Current config key.
   * @param keyPsiElement   (Optional) The corresponding key element to determine value provider by POJO property (if applicable).
   * @param valuePsiElement Current value element.
   * @param valueTextRanges Text range(s) for value text(s).
   * @param context         Context, {@link #HINT_REFERENCES_CONFIG_KEY} is made available.
   * @return References.
   */
  public abstract PsiReference @NotNull [] getValueReferences(Module module,
                                                              MetaConfigKey configKey,
                                                              @Nullable PsiElement keyPsiElement,
                                                              PsiElement valuePsiElement,
                                                              List<TextRange> valueTextRanges,
                                                              ProcessingContext context);
}
