// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.boot.application.config;

import com.intellij.lang.annotation.AnnotationHolder;
import com.intellij.lang.annotation.Annotator;
import com.intellij.lang.annotation.HighlightSeverity;
import com.intellij.microservices.config.hints.HintReferenceBase;
import com.intellij.openapi.application.ApplicationManager;
import com.intellij.openapi.editor.DefaultLanguageHighlighterColors;
import com.intellij.openapi.editor.colors.TextAttributesKey;
import com.intellij.openapi.util.TextRange;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiReference;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.JavaClassReference;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.PsiPackageReference;
import com.intellij.spring.spi.SpringSpiClassesListJamConverter;
import com.intellij.ui.SimpleTextAttributes;
import com.intellij.util.containers.ContainerUtil;
import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import it.unimi.dsi.fastutil.ints.IntSet;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public abstract class SpringBootConfigFileAnnotatorBase implements Annotator {

  /**
   * @see SpringBootPlaceholderReference
   */
  @Nullable
  protected abstract TextAttributesKey getPlaceholderTextAttributesKey();

  /**
   * @see SpringBootReplacementTokenResolver
   */
  @Nullable
  protected TextAttributesKey getReplacementTokenTextAttributesKey() {
    return DefaultLanguageHighlighterColors.METADATA;
  }

  /**
   * Highlights references in value using language default colors.
   *
   * @param element Value element.
   * @param holder  Holder.
   */
  protected void annotateValue(@NotNull PsiElement element, @NotNull AnnotationHolder holder) {
    final int elementOffset = element.getNode().getStartOffset();
    final PsiReference[] references = element.getReferences();

    boolean highlightOnlyPlaceholders =
      ContainerUtil.findInstance(references, SpringBootPlaceholderReference.class) != null;

    IntSet annotatedOffsets = new IntOpenHashSet();

    for (PsiReference reference : references) {
      TextAttributesKey key = null;
      if (highlightOnlyPlaceholders) {
        if (reference instanceof SpringBootPlaceholderReference) {
          key = getPlaceholderTextAttributesKey();
        }
      }
      else {
        if (reference instanceof JavaClassReference ||
            reference instanceof SpringSpiClassesListJamConverter.SpringSpiClassReference ||
            reference instanceof PsiPackageReference) {
          if (reference.resolve() != null) {   // FQN references are injected by default in .properties
            key = DefaultLanguageHighlighterColors.CLASS_REFERENCE;
          }
        }
        else if (reference instanceof SpringBootReplacementTokenResolver.ReplacementTokenReference) {
          key = getReplacementTokenTextAttributesKey();
        }
        else if (reference instanceof HintReferenceBase) {
          key = ((HintReferenceBase)reference).getTextAttributesKey();
        }
      }

      if (key != null) {
        TextRange highlightTextRange = reference.getRangeInElement().shiftRight(elementOffset);
        if (!annotatedOffsets.add(highlightTextRange.getStartOffset())) continue;

        doAnnotate(holder, highlightTextRange, key);
      }
    }
  }

  private static final boolean DEBUG_MODE = ApplicationManager.getApplication().isUnitTestMode();

  protected static void doAnnotate(AnnotationHolder holder, TextRange range, TextAttributesKey key) {
    if (range.isEmpty()) return;

    (DEBUG_MODE ? holder.newAnnotation(HighlightSeverity.INFORMATION, key.getExternalName()) : holder.newSilentAnnotation(HighlightSeverity.INFORMATION))
    .range(range).textAttributes(key).create();
  }

  protected static void doAnnotateEnforced(AnnotationHolder holder, TextRange range, SimpleTextAttributes key, String debugMessage) {
    if (range.isEmpty()) return;

    //noinspection HardCodedStringLiteral
    String message = DEBUG_MODE ? debugMessage : null;
    (message != null ? holder.newAnnotation(HighlightSeverity.INFORMATION, message) : holder.newSilentAnnotation(HighlightSeverity.INFORMATION))
    .range(range).enforcedTextAttributes(key.toTextAttributes()).create();
  }
}

