/*
 * Copyright 2000-2005 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jetbrains.idea.tomcat;

import com.intellij.javaee.appServers.context.DeploymentModelContext;
import com.intellij.javaee.deployment.DeploymentSource;
import com.intellij.javaee.run.configuration.CommonModel;
import com.intellij.openapi.util.io.FileUtil;
import com.intellij.openapi.util.registry.Registry;
import com.intellij.openapi.util.text.StringUtil;
import org.jdom.Element;
import org.jetbrains.idea.tomcat.server.TomcatServerModel;

public class TomcatModuleDeploymentModel extends TomcatDeploymentModelBase implements DeploymentModelContext {
  public String CONTEXT_PATH = "/";

  public TomcatModuleDeploymentModel(CommonModel parentConfiguration, DeploymentSource source) {
    super(parentConfiguration, source);

    if (!isEEArtifact() && !Registry.is("tomcat.root.context.by.default", false)) {
      CONTEXT_PATH = getContextPathBasedOnDeploymentSource();
    }
  }

  @Override
  public String getContextPath() {
    if (isEEArtifact()) {
      return getContextPathBasedOnDeploymentSource();
    }
    else {
      return getContextRoot();
    }
  }

  private String getContextPathBasedOnDeploymentSource() {
    return "/" + FileUtil.sanitizeFileName(getDeploymentSource().getPresentableName());
  }

  @Override
  public TomcatServerModel getServerModel() {
    return (TomcatServerModel)super.getServerModel();
  }

  @Override
  public boolean isDefaultContextRoot() {
    return isEEArtifact();
  }

  @Override
  public String getContextRoot() {
    String result = CONTEXT_PATH;
    Element contextRoot = TomcatUtil.findContextInContextXmlByFacet(this);
    if (StringUtil.isEmpty(result) && contextRoot != null && !getServerModel().isTomEE()) {
      result = contextRoot.getAttributeValue(TomcatContexts.PATH_ATTR);
    }
    if (!StringUtil.startsWithChar(result, '/')) {
      result = "/" + result;
    }
    if ("/".equals(result)) {
      result = "";
    }
    return result;
  }
}
