/*
 * Copyright 2000-2005 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jetbrains.idea.tomcat;

import com.intellij.execution.ExecutionException;
import com.intellij.execution.configurations.RuntimeConfigurationException;
import com.intellij.javaee.deployment.DeploymentModel;
import com.intellij.javaee.deployment.DeploymentSource;
import com.intellij.javaee.run.configuration.CommonModel;
import com.intellij.openapi.options.SettingsEditor;
import com.intellij.openapi.util.NlsSafe;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.idea.tomcat.server.TomcatLocalModel;
import org.jetbrains.idea.tomcat.server.TomcatServerModel;

import javax.swing.*;

public class TomcatDeploymentSettingsEditor extends SettingsEditor<DeploymentModel> {
  private JPanel myPanel;
  private JComboBox<String> myContextPath;
  private JPanel myInnerPanel;

  private boolean myEEArtifact = false;

  public TomcatDeploymentSettingsEditor(final CommonModel configuration, final DeploymentSource source) {
    super(() -> new TomcatModuleDeploymentModel(configuration, source));
  }

  @Override
  public void resetEditorFrom(@NotNull DeploymentModel settings) {
    final TomcatModuleDeploymentModel deploymentModel = (TomcatModuleDeploymentModel)settings;

    myEEArtifact = deploymentModel.isEEArtifact();
    myInnerPanel.setVisible(!myEEArtifact);

    if (myEEArtifact) {
      return;
    }

    final TomcatServerModel configuration = (TomcatServerModel)settings.getServerModel();

    if (configuration instanceof TomcatLocalModel) {
      updateContextPaths((TomcatLocalModel)configuration);
    }
    setSelectedContextPath(deploymentModel.CONTEXT_PATH);
  }

  @Override
  public void applyEditorTo(@NotNull DeploymentModel settings) {
    if (myEEArtifact) {
      return;
    }

    ((TomcatModuleDeploymentModel)settings).CONTEXT_PATH = getSelectedContextPath();
  }

  @Override
  @NotNull
  public JComponent createEditor() {
    return myPanel;
  }

  private String getSelectedContextPath() {
    final String item = (String)myContextPath.getEditor().getItem();

    return (item != null) ? item.trim() : "";
  }

  private void setSelectedContextPath(@NlsSafe String contextPath) {
    final int itemCount = myContextPath.getItemCount();

    for (int idx = 0; idx < itemCount; idx++) {
      if (contextPath.equals(myContextPath.getItemAt(idx))) {
        myContextPath.setSelectedIndex(idx);

        return;
      }
    }

    myContextPath.addItem(contextPath);
    myContextPath.setSelectedItem(contextPath);
  }

  private void updateContextPaths(TomcatLocalModel configuration) {
    final String selectedContextPath = getSelectedContextPath();

    myContextPath.removeAllItems();

    try {
      final TomcatContexts tomcatContexts = new TomcatContexts(configuration, configuration.getSourceBaseDirectoryPath());

      for (@NlsSafe String path : tomcatContexts.getContextPaths()) {
        myContextPath.addItem(path);
      }
    }
    catch (RuntimeConfigurationException | ExecutionException e) {
      // ignore
    }

    setSelectedContextPath(selectedContextPath);
  }
}
