package org.jetbrains.idea.tomcat;

import com.intellij.execution.ExecutionException;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.util.NlsSafe;
import com.intellij.openapi.util.io.FileUtilRt;
import com.intellij.openapi.util.text.StringUtil;
import org.jdom.Element;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.idea.tomcat.server.TomcatLocalModel;

import java.io.File;
import java.util.*;

public class TomcatContexts {

  private static final Logger LOG = Logger.getInstance(TomcatContexts.class);

  @NonNls public static final String CONTEXT_ELEMENT_NAME = "Context";
  @NonNls public static final String PATH_ATTR = "path";
  @NonNls public static final String DOC_BASE_ATTR = "docBase";
  @NonNls public static final String ROOT_DIR_NAME = "ROOT";

  @NonNls private static final String XML_EXTENSION = "xml";
  @NonNls private static final String CONTEXT_PATH_SEPARATOR = "/";
  @NonNls private static final String CONTEXT_PATH_SEPARATOR_IN_FILE_NAME = "#";

  private final TomcatServerXmlWrapper myServerXml;
  private final Element myLocalHost;
  private final List<TomcatContextItem> myItems;

  public TomcatContexts(TomcatLocalModel tomcatModel, String baseDirPath) throws ExecutionException {
    myServerXml = new TomcatServerXmlWrapper(baseDirPath);
    myItems = new ArrayList<>();
    myLocalHost = myServerXml.findLocalHost();

    final List<Element> contextElements = myLocalHost.getChildren(CONTEXT_ELEMENT_NAME);

    for (final Element contextElement : contextElements) {
      myItems.add(new TomcatContextItem(contextElement, contextElement.getAttributeValue(PATH_ATTR)) {

        @Override
        public void remove() throws ExecutionException {
          myLocalHost.removeContent(contextElement);
          myServerXml.save();
        }
      });
    }

    if (tomcatModel != null && tomcatModel.isVersion5OrHigher()) {
      final File hostDir = new File(TomcatUtil.hostDir(baseDirPath));
      final File[] contextFiles = hostDir.listFiles();

      if (contextFiles != null) {
        for (final File contextFile : contextFiles) {
          if (FileUtilRt.extensionEquals(contextFile.getName(), XML_EXTENSION)) {
            final Element rootElement;

            try {
              rootElement = TomcatUtil.loadXMLFile(contextFile);
            }
            catch (ExecutionException e) {
              LOG.info(e);

              continue;
            }

            if (CONTEXT_ELEMENT_NAME.equals(rootElement.getName())) {
              String path = FileUtilRt.getNameWithoutExtension(contextFile.getName());

              if (ROOT_DIR_NAME.equals(path)) {
                path = "";
              }

              path = CONTEXT_PATH_SEPARATOR + path.replace(CONTEXT_PATH_SEPARATOR_IN_FILE_NAME, CONTEXT_PATH_SEPARATOR);

              myItems.add(new TomcatContextItem(rootElement, path) {
                @Override
                public void remove() throws ExecutionException {
                  if (!contextFile.delete()) {
                    throw new ExecutionException(TomcatBundle.message("exception.text.cannot.delete.file", contextFile.getAbsolutePath()));
                  }
                }
              });
            }
          }
        }
      }
    }
  }

  public List<TomcatContextItem> getItems() {
    return myItems;
  }

  @Nullable
  public Element findContextByPath(String contextPath) {
    if (StringUtil.isEmpty(contextPath)) {
      contextPath = CONTEXT_PATH_SEPARATOR;
    }

    for (final TomcatContextItem item : myItems) {
      if (contextPath.equals(item.getPath())) {
        return item.getElement();
      }
    }

    return null;
  }

  public Collection<@NlsSafe String> getContextPaths() {
    final Set<String> result = new HashSet<>();

    for (final TomcatContextItem item : myItems) {
      @NlsSafe final String contextPath = item.getPath();

      if (contextPath != null) {
        result.add(contextPath);
      }
    }

    return result;
  }

  public void addOrRemoveContextElementInServerXml(String contextPath, @Nullable Element newContext) throws ExecutionException {
    final Element oldContext = findContextByPath(contextPath);

    if (oldContext != null) {
      myLocalHost.removeContent(oldContext);
    }

    if (newContext != null) {
      myLocalHost.addContent(newContext.clone());
    }

    myServerXml.save();
  }
}
