/*
 * Copyright 2000-2007 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.intellij.jam.view.tree;

import com.intellij.ide.DataManager;
import com.intellij.ide.DeleteProvider;
import com.intellij.ide.util.treeView.NodeDescriptor;
import com.intellij.ide.util.treeView.ValidateableNode;
import com.intellij.jam.JavaJamBundle;
import com.intellij.openapi.actionSystem.DataProvider;
import com.intellij.openapi.actionSystem.PlatformDataKeys;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.NlsSafe;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.ui.treeStructure.SimpleNode;
import com.intellij.ui.treeStructure.SimpleTree;
import com.intellij.util.OpenSourceUtil;
import org.jetbrains.annotations.Nls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;
import java.awt.*;
import java.awt.event.InputEvent;

/**
 * author: lesya
 */
public abstract class JamNodeDescriptor<P> extends SimpleNode implements DataProvider, ValidateableNode {
  public static final JamNodeDescriptor[] EMPTY_ARRAY = new JamNodeDescriptor[0];

  private final Object myParameters;
  private final P myElement;
  @Nls private String myTooltip;

  protected JamNodeDescriptor(Project project, NodeDescriptor parentDescriptor, Object parameters, P element) {
    super(project, parentDescriptor);
    myParameters = parameters;
    myElement = element;
  }

  @Override
  public Object @NotNull [] getEqualityObjects() {
    return new Object[] { myElement };
  }

  @Override
  protected void doUpdate() {
    setUniformIcon(getNewIcon());
    final String nodeText = getNewNodeText();
    setNodeText(StringUtil.isNotEmpty(nodeText) ? nodeText : JavaJamBundle.message("unnamed.element.presentable.name"), null, !isValid());
    myTooltip = getNewTooltip();
  }

  @Override
  protected P updateElement() {
    return myElement;
  }

  @Nullable  @NlsSafe
  protected abstract String getNewNodeText();

  @Nullable
  protected Icon getNewIcon() {
    return getIcon();
  }

  protected Color getNewColor() {
    return myColor;
  }

  @Override
  public SimpleNode @NotNull [] getChildren(){
    return EMPTY_ARRAY;
  }

  @Override
  @Nullable
  public Object getData(@NotNull String dataId) {
    return null;
  }

  @Nls
  public String getTooltip() {
    return myTooltip;
  }

  @Nullable @Nls
  public String getNewTooltip() {
    return getNewNodeText();
  }

  @Override
  public int getWeight() {
    return 0;
  }

  @Nullable
  protected DeleteProvider getDeleteProvider() {
    return null;
  }

  protected Object getParameters() {
    return myParameters;
  }

  @Nullable
  public final Object getDataForElement(String dataId) {
    if (PlatformDataKeys.DELETE_ELEMENT_PROVIDER.is(dataId)) {
      return getDeleteProvider();
    }
    return getData(dataId);
  }

  public final P getJamElement() {
    return myElement;
  }

  @Nullable
  public String getComment() {
    return null;
  }

  @Override
  public boolean isValid() {
    return true;
  }

  @Override
  public void handleDoubleClickOrEnter(final SimpleTree tree, final InputEvent inputEvent) {
    OpenSourceUtil.openSourcesFrom(DataManager.getInstance().getDataContext(tree), false);
  }
}
