package com.intellij.jam;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.project.DumbService;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.roots.ProjectRootManager;
import com.intellij.psi.JavaPsiFacade;
import com.intellij.psi.util.CachedValuesManager;
import com.intellij.util.containers.ConcurrentFactoryMap;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Map;
import java.util.concurrent.ConcurrentMap;

import static com.intellij.psi.search.GlobalSearchScope.allScope;
import static com.intellij.psi.search.GlobalSearchScope.moduleWithDependenciesAndLibrariesScope;
import static com.intellij.psi.util.CachedValueProvider.Result;
import static java.util.Collections.emptyMap;

public final class JavaLibraryUtils {
  /**
   * Checks if the passed library class is available in the project. Should be used only with constant class names from a known library.
   * Returns false from dumb mode or if the project is already disposed.
   */
  public static boolean hasLibraryClass(@Nullable Project project, @NotNull String classFqn) {
    if (project == null || project.isDisposed()) return false;
    return getLibraryClassMap(project).getOrDefault(classFqn, false);
  }

  /**
   * Checks if the passed library class is available in the module. Should be used only with constant class names from a known library.
   * Returns false from dumb mode or if the module is already disposed.
   */
  public static boolean hasLibraryClass(@Nullable Module module, @NotNull String classFqn) {
    if (module == null || module.isDisposed()) return false;
    return getLibraryClassMap(module).getOrDefault(classFqn, false);
  }

  private static Map<String, Boolean> getLibraryClassMap(@NotNull Project project) {
    if (DumbService.isDumb(project)) return emptyMap();

    return CachedValuesManager.getManager(project).getCachedValue(project, () -> {
      ConcurrentMap<String, Boolean> map = ConcurrentFactoryMap.createMap(classFqn -> {
        return JavaPsiFacade.getInstance(project).findClass(classFqn, allScope(project)) != null;
      });
      return Result.createSingleDependency(map, ProjectRootManager.getInstance(project));
    });
  }

  private static Map<String, Boolean> getLibraryClassMap(@NotNull Module module) {
    if (DumbService.isDumb(module.getProject())) return emptyMap();

    return CachedValuesManager.getManager(module.getProject()).getCachedValue(module, () -> {
      ConcurrentMap<String, Boolean> map = ConcurrentFactoryMap.createMap(classFqn -> {
        return JavaPsiFacade.getInstance(module.getProject()).findClass(classFqn, moduleWithDependenciesAndLibrariesScope(module)) != null;
      });
      return Result.createSingleDependency(map, ProjectRootManager.getInstance(module.getProject()));
    });
  }
}
