/*
 * Copyright 2000-2014 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.jam;

import com.intellij.jam.model.util.JamCommonUtil;
import com.intellij.jam.reflect.JamAnnotationMeta;
import com.intellij.jam.reflect.JamMemberMeta;
import com.intellij.openapi.components.Service;
import com.intellij.openapi.project.Project;
import com.intellij.psi.*;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.semantic.SemKey;
import com.intellij.semantic.SemService;
import com.intellij.util.BitUtil;
import com.intellij.util.Function;
import com.intellij.util.Processor;
import com.intellij.util.containers.ContainerUtil;
import org.intellij.lang.annotations.MagicConstant;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.lang.annotation.ElementType;
import java.lang.annotation.Target;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Service(Service.Level.PROJECT)
public final class JamService {
  private static final Map<SemKey, SemKey<JamMemberMeta>> ourMetaKeys = new HashMap<>();

  public static final SemKey<JamMemberMeta> MEMBER_META_KEY = SemKey.createKey("JamMemberMeta");
  public static final SemKey<JamAnnotationMeta> ANNO_META_KEY = SemKey.createKey("JamAnnotationMeta");
  public static final SemKey<JamElement> JAM_ELEMENT_KEY = SemKey.createKey("JamElement");
  public static final SemKey<JamElement> JAM_ALIASING_ELEMENT_KEY = SemKey.createKey("JamAliasingElement");

  public static final SemKey<JamMemberMeta> ALIASING_MEMBER_META_KEY = getMetaKey(JAM_ALIASING_ELEMENT_KEY);
  private final PsiManager myPsiManager;
  private final SemService mySemService;

  private JamService(@NotNull Project project) {
    myPsiManager = PsiManager.getInstance(project);
    mySemService = SemService.getSemService(project);
  }

  public static JamService getJamService(@NotNull Project project) {
    return project.getService(JamService.class);
  }

  private static void processMembers(PsiClass psiClass, @Flags int flags, Processor<? super PsiMember> processor) {
    if (BitUtil.isSet(flags, CHECK_CLASS)) {
      if (BitUtil.isSet(flags, CHECK_DEEP)) {
        for (PsiClass curClass : JamCommonUtil.getSuperClassList(psiClass)) {
          processor.process(curClass);
        }
      }
      else {
        processor.process(psiClass);
      }
    }
    if (BitUtil.isSet(flags, CHECK_METHOD)) {
      ContainerUtil.process(BitUtil.isSet(flags, CHECK_DEEP) ? psiClass.getAllMethods() : psiClass.getMethods(), processor);
    }
    if (BitUtil.isSet(flags, CHECK_FIELD)) {
      ContainerUtil.process(BitUtil.isSet(flags, CHECK_DEEP) ? psiClass.getAllFields() : psiClass.getFields(), processor);
    }
  }

  @Nullable
  public <T extends JamElement> T getJamElement(@NotNull PsiElement psi,
                                                JamMemberMeta<? extends PsiModifierListOwner, ? extends T>... metas) {
    for (JamMemberMeta<? extends PsiModifierListOwner, ? extends T> meta : metas) {
      final T element = mySemService.getSemElement(meta.getJamKey(), psi);
      if (element != null) {
        return element;
      }
    }
    return null;
  }

  @Nullable
  public <T extends JamElement> T getJamElement(SemKey<T> key, @NotNull PsiElement psi) {
    if (!psi.isValid()) {
      throw new PsiInvalidElementAccessException(psi);
    }
    try {
      return mySemService.getSemElement(key, psi);
    }
    catch (PsiInvalidElementAccessException e) {
      throw new RuntimeException("Element invalidated: old=" + psi + "; new=" + e.getPsiElement(), e);
    }
  }

  @NotNull
  public <T extends PsiModifierListOwner> List<JamMemberMeta> getMetas(@NotNull T psi) {
    return mySemService.getSemElements(MEMBER_META_KEY, psi);
  }

  @Nullable
  public <T extends PsiModifierListOwner> JamMemberMeta<T, ?> getMeta(@NotNull T psi, SemKey<? extends JamMemberMeta<T, ?>> key) {
    return mySemService.getSemElement(key, psi);
  }

  @Nullable
  public JamAnnotationMeta getMeta(@NotNull PsiAnnotation anno) {
    return mySemService.getSemElement(ANNO_META_KEY, anno);
  }

  public <T extends JamElement> List<T> getJamClassElements(final JamMemberMeta<? super PsiClass, ? extends T> meta,
                                                            @NonNls final String anno,
                                                            final GlobalSearchScope scope) {
    return getJamClassElements(meta.getJamKey(), anno, scope);
  }

  public <T extends JamElement> List<T> getJamMethodElements(final JamMemberMeta<? super PsiMethod, ? extends T> meta,
                                                             @NonNls final String anno,
                                                             final GlobalSearchScope scope) {
    return getJamMethodElements(meta.getJamKey(), anno, scope);
  }

  public <T extends JamElement> List<T> getJamFieldElements(final JamMemberMeta<? super PsiField, ? extends T> meta,
                                                            @NonNls final String anno,
                                                            final GlobalSearchScope scope) {
    return getJamFieldElements(meta.getJamKey(), anno, scope);
  }

  public static final int CHECK_CLASS = 0x01;
  public static final int CHECK_METHOD = 0x02;
  public static final int CHECK_FIELD = 0x04;
  public static final int CHECK_DEEP = 0x08;

  public static final int CHECK_ALL_DEEP = CHECK_CLASS | CHECK_METHOD | CHECK_FIELD | CHECK_DEEP;

  @MagicConstant(flags = {CHECK_CLASS, CHECK_METHOD, CHECK_FIELD, CHECK_DEEP, CHECK_ALL_DEEP})
  @Target({ElementType.PARAMETER, ElementType.METHOD})
  private @interface Flags { }

  public <T extends JamElement> List<T> getAnnotatedMembersList(@NotNull PsiClass psiClass,
                                                                @Flags int checkFlags,
                                                                JamMemberMeta<? extends PsiMember, ? extends T>... metas) {
    List<T> result = new ArrayList<>();
    processMembers(psiClass, checkFlags, member -> {
      for (JamMemberMeta<? extends PsiMember, ? extends T> meta : metas) {
        final T element = mySemService.getSemElement(meta.getJamKey(), member);
        ContainerUtil.addIfNotNull(result, element);
      }
      return true;
    });
    return result;
  }

  public <T extends JamElement> List<T> getJamClassElements(final SemKey<? extends T> key,
                                                            @NonNls final String anno,
                                                            final GlobalSearchScope scope) {
    final List<T> result = new ArrayList<>();
    JamCommonUtil.findAnnotatedElements(PsiClass.class, anno, myPsiManager, scope, psiMember -> {
      ContainerUtil.addIfNotNull(result, getJamElement(key, psiMember));
      return true;
    });
    return result;
  }

  public <T extends JamElement> List<T> getJamMethodElements(final SemKey<? extends T> key,
                                                             @NonNls final String anno,
                                                             final GlobalSearchScope scope) {
    final List<T> result = new ArrayList<>();
    JamCommonUtil.findAnnotatedElements(PsiMethod.class, anno, myPsiManager, scope, psiMember -> {
      ContainerUtil.addIfNotNull(result, getJamElement(key, psiMember));
      return true;
    });
    return result;
  }

  public <T extends JamElement> List<T> getJamFieldElements(final SemKey<? extends T> key,
                                                            @NonNls final String anno,
                                                            final GlobalSearchScope scope) {
    final List<T> result = new ArrayList<>();
    JamCommonUtil.findAnnotatedElements(PsiField.class, anno, myPsiManager, scope, psiMember -> {
      ContainerUtil.addIfNotNull(result, getJamElement(key, psiMember));
      return true;
    });
    return result;
  }

  public <T extends JamElement> List<T> getJamParameterElements(final SemKey<? extends T> key,
                                                                @NonNls final String anno,
                                                                final GlobalSearchScope scope) {
    final List<T> result = new ArrayList<>();
    JamCommonUtil.findAnnotatedElements(PsiParameter.class, anno, myPsiManager, scope, psiParameter -> {
      ContainerUtil.addIfNotNull(result, getJamElement(key, psiParameter));
      return true;
    });
    return result;
  }

  public <T extends JamElement> List<T> getAnnotatedMembersList(@NotNull PsiClass psiClass,
                                                                SemKey<? extends T> clazz,
                                                                @Flags int checkFlags) {
    List<T> result = new ArrayList<>();

    processMembers(psiClass, checkFlags, member -> {
      ContainerUtil.addIfNotNull(result, getJamElement(clazz, member));
      return true;
    });
    return result;
  }

  public static SemKey<JamMemberMeta> getMetaKey(final SemKey<? extends JamElement> jamKey) {
    synchronized (ourMetaKeys) {
      SemKey<JamMemberMeta> result = ourMetaKeys.get(jamKey);
      if (result == null) {
        SemKey<?>[] jamSupers = jamKey.getSupers();
        SemKey[] metaSupers = ContainerUtil.map2Array(jamSupers, SemKey.class, (Function<SemKey, SemKey<?>>)key -> {
          //noinspection unchecked
          return getMetaKey(key);
        });
        //noinspection unchecked
        ourMetaKeys.put(jamKey, result = MEMBER_META_KEY.subKey(jamKey + "Meta", metaSupers));
      }
      return result;
    }
  }
}
