// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.jam

import com.intellij.codeInsight.completion.CompletionUtil
import com.intellij.jam.reflect.JamStringAttributeMeta
import com.intellij.openapi.application.ApplicationManager
import com.intellij.openapi.diagnostic.Logger
import com.intellij.patterns.StandardPatterns
import com.intellij.patterns.uast.UExpressionPattern
import com.intellij.patterns.uast.capture
import com.intellij.patterns.uast.injectionHostUExpression
import com.intellij.patterns.uast.uExpression
import com.intellij.psi.*
import com.intellij.semantic.SemService
import org.jetbrains.uast.*

internal class JamReferenceContributor : PsiReferenceContributor() {

  override fun registerReferenceProviders(registrar: PsiReferenceRegistrar) {
    registrar.registerReferenceProviderByUsage(
      uInjectionHostOrSingleReferenceInAnnotation(),
      uastReferenceProviderByUsage { uElement, physicalPsiHost, usagePsi ->
        getReferences(uElement, physicalPsiHost, usagePsi)
      }, PsiReferenceRegistrar.HIGHER_PRIORITY
    )
  }

  @Suppress("UNCHECKED_CAST")
  private fun getReferences(uElement: UElement, physicalPsiHost: PsiLanguageInjectionHost, usagePsi: PsiElement): Array<PsiReference> {
    val usageElement = compareAndGetUElement(usagePsi, uElement) ?: return PsiReference.EMPTY_ARRAY

    val (psiAnnotation, name) = getContainingAnnotationEntry(usageElement) ?: return PsiReference.EMPTY_ARRAY

    val originalUsageElement = compareAndGetUElement(CompletionUtil.getOriginalOrSelf(usagePsi), uElement)
    val (originalPsiAnnotation, _) = getContainingAnnotationEntry(originalUsageElement)
                                     ?: return PsiReference.EMPTY_ARRAY
    val metas = SemService.getSemService(physicalPsiHost.project).getSemElements(JamService.ANNO_META_KEY, originalPsiAnnotation)

    for (annotationMeta in metas) {
      val meta = annotationMeta?.findAttribute(name) as? JamStringAttributeMeta<*, *> ?: continue
      val converter = meta.converter as JamConverter<in Any>
      val jam = meta.getJam(PsiElementRef.real(psiAnnotation))

      if (usagePsi === physicalPsiHost) {
        // Inplace references in annotation attribute
        (jam as? List<JamStringAttributeElement<*>>)?.let { list ->
          return list.flatMap { jamStringAttributeElement ->
            convertReferences(converter, physicalPsiHost, jamStringAttributeElement)
          }.toTypedArray()
        }
        (jam as? JamStringAttributeElement<*>)?.let { jamStringAttributeElement ->
          return convertReferences(converter, physicalPsiHost, jamStringAttributeElement).toTypedArray()
        }
      }
      else {
        // References by usage
        (jam as? List<JamStringAttributeElement<*>>)?.let { list ->
          for (jamStringAttributeElement in list) {
            if (usageElement.isUastChildOf(jamStringAttributeElement.psiElement.toUElement())) {
              return converter.createReferences(jamStringAttributeElement, physicalPsiHost, usagePsi)
            }
          }
          return PsiReference.EMPTY_ARRAY
        }
        (jam as? JamStringAttributeElement<*>)?.let { jamStringAttributeElement ->
          return converter.createReferences(jamStringAttributeElement, physicalPsiHost, usagePsi)
        }
      }
    }

    return PsiReference.EMPTY_ARRAY
  }

  private fun compareAndGetUElement(target: PsiElement, knownUElement: UElement): UElement? =
    if (target === knownUElement.sourcePsi)
      knownUElement // reuse UAST element
    else
      target.toUElement()

  private fun convertReferences(converter: JamConverter<in Any>,
                                physicalPsiHost: PsiLanguageInjectionHost,
                                jamStringAttributeElement: JamStringAttributeElement<*>): List<PsiReference> {
    val annotationValuePsi = jamStringAttributeElement.psiElement
    return when {
      // handling only java concatenation, suppose Kotlin will use string templates which are emulated as PsiLiterals-s
      annotationValuePsi is PsiPolyadicExpression ->
        if (annotationValuePsi.operands.contains<PsiElement>(physicalPsiHost))
          converter.createReferences(jamStringAttributeElement, physicalPsiHost, physicalPsiHost).filter { it.element == physicalPsiHost }
        else
          emptyList()

      isJamElementFromHost(annotationValuePsi, physicalPsiHost) ->
        converter.createReferences(jamStringAttributeElement, physicalPsiHost, physicalPsiHost).map { wrapRef(it, physicalPsiHost) }

      else -> emptyList()
    }
  }

  companion object {
    @JvmField
    val STRING_IN_ANNO: UExpressionPattern<*, *> =
      injectionHostUExpression().annotationParams(capture(UAnnotation::class.java), StandardPatterns.string())
  }
}

private fun uInjectionHostOrSingleReferenceInAnnotation(): UExpressionPattern.Capture<UExpression> =
  uExpression().filter {
    if (it is UReferenceExpression) {
      var uastParent = it.uastParent
      if (uastParent is UNamedExpression) {
        uastParent = uastParent.uastParent
      }

      uastParent !is UPolyadicExpression
      && getContainingAnnotationEntry(it) != null
    }
    else {
      it.isInjectionHost() && getContainingAnnotationEntry(it) != null
    }
  }

fun isJamElementFromHost(psiAnnotationMemberValue: PsiAnnotationMemberValue?, physicalPsiHost: PsiLanguageInjectionHost) =
  psiAnnotationMemberValue == physicalPsiHost ||
  psiAnnotationMemberValue.toUElementOfType<UExpression>()?.sourceInjectionHost == physicalPsiHost

fun unwrapReference(reference: PsiReference): PsiReference = when (reference) {
  is NonPhysicalReferenceWrapper -> reference.wrappedReference
  else -> reference
}

private fun wrapRef(it: PsiReference, physicalPsi: PsiElement): PsiReference =
  if (it.element === physicalPsi) it else NonPhysicalReferenceWrapper(physicalPsi, it)

private class NonPhysicalReferenceWrapper(physicalElement: PsiElement, val wrappedReference: PsiReference) :
  PsiReferenceBase<PsiElement>(physicalElement, wrappedReference.rangeInElement) {

  init {
    if (ApplicationManager.getApplication().run { isInternal || isUnitTestMode } && wrappedReference.element.isPhysical) {
      Logger.getInstance(NonPhysicalReferenceWrapper::class.java)
        .error("creating a NonPhysicalReferenceWrapper for a physical reference $wrappedReference on ${wrappedReference.element}")
    }
  }

  override fun resolve(): PsiElement? = wrappedReference.resolve()

  override fun getVariants(): Array<Any> = wrappedReference.variants

  override fun isSoft(): Boolean = wrappedReference.isSoft
}
