/*
 * Copyright 2000-2009 JetBrains s.r.o.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package com.intellij.persistence.util;

import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiType;
import org.jetbrains.annotations.Nullable;

import java.util.Arrays;

/**
* @author Gregory.Shrago
*/
public class JavaTypeInfo {
  public final JavaContainerType containerType;
  public final PsiType baseType;
  @Nullable private final PsiClass myConverterClass;
  public final PsiType[] parameters;

  public JavaTypeInfo(JavaContainerType containerType, PsiType baseType, PsiType... parameters) {
    this(containerType, baseType, null, parameters);
  }

  public JavaTypeInfo(JavaContainerType containerType, PsiType baseType, @Nullable PsiClass converterClass, PsiType... parameters) {
    this.containerType = containerType;
    this.baseType = baseType;
    myConverterClass = converterClass;
    this.parameters = parameters;
  }

  @Nullable
  public PsiType getValueType() {
    if (containerType == null || parameters.length == 0) return baseType;
    return parameters[0];
  }

  @Nullable
  public PsiClass getConverterClass() {
    return myConverterClass;
  }

  @Nullable
  public PsiType getKeyType() {
    if (containerType == null) return null;
    if (containerType == JavaContainerType.MAP) return parameters.length == 2 ? parameters[1] : null;
    return null;
  }

  public JavaTypeInfo copy() {
    return new JavaTypeInfo(containerType, baseType, parameters.clone());
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    JavaTypeInfo that = (JavaTypeInfo)o;

    if (baseType != null ? !baseType.equals(that.baseType) : that.baseType != null) return false;
    if (containerType != that.containerType) return false;
    if (!Arrays.equals(parameters, that.parameters)) return false;

    return true;
  }

  @Override
  public int hashCode() {
    int result = containerType != null ? containerType.hashCode() : 0;
    result = 31 * result + (baseType != null ? baseType.hashCode() : 0);
    result = 31 * result + Arrays.hashCode(parameters);
    return result;
  }
}
