/*
 * Copyright (c) 2000-2006 JetBrains s.r.o. All Rights Reserved.
 */

package com.intellij.persistence.model;

import com.intellij.jam.model.common.CommonModelElement;
import com.intellij.jam.view.JamDeleteHandler;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.persistence.facet.PersistenceFacet;
import com.intellij.persistence.util.PersistenceCommonUtil;
import com.intellij.persistence.util.PersistenceModelBrowser;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElement;
import com.intellij.util.Processor;
import com.intellij.util.Query;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.xml.GenericDomValue;
import com.intellij.util.xml.GenericValue;

import java.util.Collection;

public class JpaPersistenceObjectDeleteHandler extends JamDeleteHandler {

  @Override
  public void addPsiElements(final CommonModelElement element, final Collection<? super PsiElement> result) {
    if (element instanceof PersistentObject) {
      final PsiClass psiClass = ((PersistentObject)element).getClazz().getValue();
      final Module module = psiClass == null ? null : ModuleUtilCore.findModuleForPsiElement(psiClass);
      ContainerUtil.addIfNotNull(result, psiClass);
      if (module != null) {
        for (PersistenceFacet facet : PersistenceCommonUtil.getAllPersistenceFacetsWithDependencies(module)) {
          for (PersistencePackage unit : facet.getPersistenceUnits()) {
            for (GenericValue<PsiClass> classValue : unit.getModelHelper().getClasses()) {
              if (psiClass.equals(classValue.getValue())) {
                if (classValue instanceof GenericDomValue) {
                  ContainerUtil.addIfNotNull(result, ((GenericDomValue)classValue).getXmlTag());
                }
              }
            }
          }
        }
      }
    }
  }

  @Override
  public void addModelElements(final CommonModelElement element, final Collection<CommonModelElement> result) {
    if (!(element instanceof PersistentObject) || !element.isValid()) return;
    final PersistentObject persistentObject = (PersistentObject)element;
    final PsiClass clazz = persistentObject.getClazz().getValue();
    final Module module = clazz == null? null : ModuleUtilCore.findModuleForPsiElement(clazz);
    if (module == null) return;
    final Processor<PersistentObject> processor = object -> !clazz.getManager().areElementsEquivalent(clazz, object.getClazz().getValue());
    for (PersistenceFacet facet : PersistenceCommonUtil.getAllPersistenceFacetsWithDependencies(module)) {
      for (PersistencePackage unit : facet.getPersistenceUnits()) {
        final PersistenceModelBrowser browser = PersistenceCommonUtil.createFacetAndUnitModelBrowser(facet, unit, null);
        final PersistenceMappings mappings = facet.getEntityMappings(unit);
        for (PersistentObject object : PersistenceCommonUtil.queryPersistentObjects(mappings)) {
          if (!processor.process(object)) {
            result.add(object);
            continue;
          }
          for (PersistentAttribute attribute : browser.queryAttributes(object)) {
            final Query<PersistentObject> query;
            if (attribute instanceof PersistentEmbeddedAttribute) {
              query = browser.queryTargetPersistentObjects((PersistentEmbeddedAttribute)attribute);
            }
            else if (attribute instanceof PersistentRelationshipAttribute) {
              query = browser.queryTargetPersistentObjects((PersistentRelationshipAttribute)attribute);
            }
            else continue;
            if (!query.forEach(processor)) {
              result.add(attribute);
            }
          }
        }
      }
    }
  }

}
