/*
 * Copyright 2000-2007 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.persistence.diagram;

import com.intellij.codeInsight.NullableNotNullManager;
import com.intellij.facet.Facet;
import com.intellij.facet.FacetModificationTrackingService;
import com.intellij.ide.util.PsiNavigationSupport;
import com.intellij.jam.model.common.CommonModelElement;
import com.intellij.jam.model.util.JamCommonUtil;
import com.intellij.jam.view.DefaultUserResponse;
import com.intellij.jam.view.JamDeleteProvider;
import com.intellij.openapi.actionSystem.CommonDataKeys;
import com.intellij.openapi.actionSystem.DataProvider;
import com.intellij.openapi.actionSystem.LangDataKeys;
import com.intellij.openapi.actionSystem.PlatformDataKeys;
import com.intellij.openapi.editor.EditorFactory;
import com.intellij.openapi.editor.event.DocumentEvent;
import com.intellij.openapi.editor.event.DocumentListener;
import com.intellij.openapi.graph.builder.GraphBuilder;
import com.intellij.openapi.graph.view.EditMode;
import com.intellij.openapi.graph.view.Graph2DView;
import com.intellij.openapi.util.ModificationTracker;
import com.intellij.openapi.util.ModificationTrackerListener;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.persistence.PersistenceDataKeys;
import com.intellij.persistence.facet.PersistenceFacet;
import com.intellij.persistence.model.*;
import com.intellij.persistence.util.PersistenceCommonUtil;
import com.intellij.persistence.util.PersistenceModelBrowser;
import com.intellij.pom.Navigatable;
import com.intellij.psi.*;
import com.intellij.psi.util.PsiUtilCore;
import com.intellij.util.PairProcessor;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.xml.ElementPresentationManager;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;
import java.util.*;

/**
 * @author Gregory.Shrago
 */
public class DefaultDiagramSupport implements PersistenceDiagramSupport<PersistencePackage, PersistentObject, PersistentAttribute> {
  private Map<PsiClass, PersistentObject> myClassMap;
  private Collection<PersistentObject> myPersistentObjects;
  private final PersistenceFacet myFacet;
  private PersistenceModelBrowser myModelBrowser;

  public DefaultDiagramSupport(final PersistenceFacet facet) {
    myFacet = facet;
  }

  public PersistenceFacet getFacet() {
    return myFacet;
  }

  @Override
  public ModificationTracker getModificationTracker(final PersistencePackage unit) {
    return myFacet.getModificationTracker();
  }

  @Override
  public void startDataModelUpdate(final PersistencePackage persistencePackage) {
    myModelBrowser = PersistenceCommonUtil.createFacetAndUnitModelBrowser(myFacet, persistencePackage, null);
    if (persistencePackage == null || !persistencePackage.isValid()) {
      myPersistentObjects = Collections.emptyList();
      myClassMap = Collections.emptyMap();
    }
    else {
      final PersistenceMappings entityMappings = myFacet.getEntityMappings(persistencePackage);
      startDataModelUpdate(entityMappings);
    }
  }

  public void startDataModelUpdate(final PersistenceMappings entityMappings) {
    myPersistentObjects = PersistenceCommonUtil.queryPersistentObjects(entityMappings).findAll();
    myClassMap = new HashMap<>();
    for (PersistentObject persistentObject : myPersistentObjects) {
      final PsiClass psiClass = persistentObject.getClazz().getValue();
      if (psiClass != null) {
        myClassMap.put(psiClass, persistentObject);
      }
    }
  }


  @Override
  public void finishDataModelUpdate() {
    myPersistentObjects = null;
    myClassMap = null;
  }


  public PersistenceModelBrowser getModelBrowser() {
    return myModelBrowser;
  }

  @Override
  public void processEntities(final PairProcessor<? super PersistentObject, String> pairProcessor, final boolean superclasses,
                              final boolean embeddables) {
    for (PersistentObject object : myPersistentObjects) {
      if (!superclasses && object instanceof PersistentSuperclass ||
          !embeddables && object instanceof PersistentEmbeddable) continue;
      final PsiClass psiClass = object.getClazz().getValue();
      if (!pairProcessor.process(object, psiClass == null? object.getClazz().getStringValue() : psiClass.getQualifiedName())) {
        return;
      }
    }
  }

  @Override
  public void processSuper(final PersistentObject sourceEntity, final PairProcessor<? super PersistentObject, String> pairProcessor) {
    final PsiClass psiClass = sourceEntity.getClazz().getValue();
    if (psiClass == null) return;
    for (final PsiClass curClass : JamCommonUtil.getSuperClassList(psiClass.getSuperClass())) {
      final PersistentObject superObject = myClassMap.get(curClass);
      if (superObject != null && !pairProcessor.process(superObject, curClass.getQualifiedName())) {
        return;
      }
    }
  }

  @Override
  public void processRelated(final PersistentObject sourceEntity, final PairProcessor<? super PersistentAttribute, String> pairProcessor) {
    for (PersistentAttribute persistentAttribute : sourceEntity.getObjectModelHelper().getAttributes()) {
      if (persistentAttribute instanceof PersistentRelationshipAttribute) {
        final PersistentRelationshipAttribute attribute = (PersistentRelationshipAttribute)persistentAttribute;
        final PsiClass psiClass = PersistenceCommonUtil.getTargetClass(attribute);
        if (!pairProcessor.process(persistentAttribute, psiClass == null? attribute.getTargetEntityClass().getStringValue() : psiClass.getQualifiedName())) {
          return;
        }
      }
    }
  }

  @Override
  public void processEmbedded(final PersistentObject sourceEntity, final PairProcessor<? super PersistentAttribute, String> pairProcessor) {
    for (PersistentAttribute persistentAttribute : sourceEntity.getObjectModelHelper().getAttributes()) {
      if (persistentAttribute instanceof PersistentEmbeddedAttribute) {
        final PersistentEmbeddedAttribute attribute = (PersistentEmbeddedAttribute)persistentAttribute;
        final PsiClass psiClass = PersistenceCommonUtil.getTargetClass(attribute);
        if (!pairProcessor.process(persistentAttribute, psiClass == null ? attribute.getTargetEmbeddableClass().getStringValue() : psiClass.getQualifiedName())) {
          return;
        }
      }
    }
  }

  @Override
  public void processAttributes(final PersistentObject persistentObject, final PairProcessor<? super PersistentAttribute, String> pairProcessor) {
    for (PersistentAttribute persistentAttribute : persistentObject.getObjectModelHelper().getAttributes()) {
      if (persistentAttribute instanceof PersistentTransientAttribute ||
          persistentAttribute instanceof PersistentRelationshipAttribute ||
          persistentAttribute instanceof PersistentEmbeddedAttribute) {
        continue;
      }
      if (!pairProcessor.process(persistentAttribute, persistentAttribute.getName().getValue())) {
        return;
      }
    }
  }

  @Override
  @Nullable
  public PersistentObject getAttributeTarget(final PersistentAttribute persistentAttribute) {
    if (persistentAttribute instanceof PersistentRelationshipAttribute) {
      return myModelBrowser.queryTargetPersistentObjects((PersistentRelationshipAttribute)persistentAttribute).findFirst();
    }
    else if (persistentAttribute instanceof PersistentEmbeddedAttribute) {
      return myModelBrowser.queryTargetPersistentObjects((PersistentEmbeddedAttribute)persistentAttribute).findFirst();
    }
    return null;
  }

  @Override
  public String getUniqueId(final PersistentObject persistentObject) {
    return PersistenceCommonUtil.getUniqueId(persistentObject == null? null : persistentObject.getIdentifyingPsiElement());
  }

  @Override
  @Nullable
  public PersistentAttribute getInverseSideAttribute(final PersistentAttribute persistentAttribute) {
    assert persistentAttribute instanceof PersistentRelationshipAttribute;
    return myModelBrowser.queryTheOtherSideAttributes((PersistentRelationshipAttribute)persistentAttribute, false).findFirst();
  }


  @Override
  @Nullable
  public String getAttributeName(final PersistentAttribute persistentAttribute) {
    return persistentAttribute.getName().getValue();
  }

  @Override
  @Nullable
  public PsiType getAttributePsiType(final PersistentAttribute persistentAttribute) {
    return persistentAttribute.getPsiType();
  }

  @Override
  @NonNls
  @NotNull
  public String getEntityTypeName(final PersistentObject persistentObject) {
    return ElementPresentationManager.getTypeNameForObject(persistentObject);
  }

  @Override
  @NonNls
  @NotNull
  public String getAttributeTypeName(final PersistentAttribute persistentAttribute) {
    return ElementPresentationManager.getTypeNameForObject(persistentAttribute);
  }

  @Override
  public boolean isIdAttribute(final PersistentAttribute persistentAttribute) {
    return persistentAttribute.getAttributeModelHelper().isIdAttribute();
  }

  @Override
  @NotNull
  public String getAttributeMultiplicityLabel(final PersistentAttribute first, final PersistentAttribute second, final boolean isSource) {
    assert first instanceof PersistentRelationshipAttribute && (second == null || second instanceof PersistentRelationshipAttribute);
    final PersistentRelationshipAttribute attribute = ((PersistentRelationshipAttribute)first);
    final boolean many = attribute.getAttributeModelHelper().getRelationshipType().isMany(isSource);
    final boolean optional = attribute.getAttributeModelHelper().isRelationshipSideOptional(isSource) && !isNotNull((PersistentRelationshipAttribute)second);
    return PersistenceCommonUtil.getMultiplicityString(optional, many);
  }

  private boolean isNotNull(final PersistentRelationshipAttribute attribute) {
    final PsiMember owner = attribute == null? null : attribute.getPsiMember();
    return owner != null && NullableNotNullManager.isNotNull(owner);
  }

  @Override
  @Nullable
  public Icon getEntityIcon(final PersistentObject persistentObject) {
    return ElementPresentationManager.getIcon(persistentObject);
  }

  @Override
  @Nullable
  public Icon getAttributeIcon(final PersistentAttribute persistentAttribute, final boolean forceId) {
    return ElementPresentationManager.getIcon(persistentAttribute);
  }

  @Override
  public @NotNull DataProvider createDataProvider(final PersistenceDiagram<PersistencePackage, PersistentObject, PersistentAttribute> diagram) {
    return dataId -> getData(diagram, dataId);
  }

  protected @Nullable Object getData(@NotNull PersistenceDiagram<PersistencePackage, PersistentObject, PersistentAttribute> diagram,
                                     @NotNull String dataId) {
    if (CommonDataKeys.PROJECT.is(dataId)) {
      return diagram.getProject();
    }
    else if (PersistenceDataKeys.PERSISTENCE_FACET.is(dataId)) {
      return myFacet;
    }
    else if (PersistenceDataKeys.PERSISTENCE_UNIT.is(dataId)) {
      return diagram.getUnit();
    }
    else if (!diagram.getUnit().isValid()) {
      return null;
    }
    else if (LangDataKeys.MODULE.is(dataId)) {
      return diagram.getUnit().getModule();
    }
    else if (PersistenceDataKeys.MODEL_ELEMENT_CONTEXT.is(dataId)) {
      return diagram.getSelectedEntity() != null ? diagram.getSelectedEntity() : diagram.getUnit();
    }
    else if (LangDataKeys.PSI_ELEMENT_ARRAY.is(dataId)) {
      Collection<CommonModelElement> elements = getSelectedElements(diagram);
      if (!elements.isEmpty()) {
        final HashSet<PsiElement> result = new HashSet<>();
        for (CommonModelElement element : elements) {
          final PsiElement psiElement = element.getIdentifyingPsiElement();
          ContainerUtil.addIfNotNull(result, psiElement);
        }
        return PsiUtilCore.toPsiElementArray(result);
      }
      return null;
    }
    else if (PlatformDataKeys.DELETE_ELEMENT_PROVIDER.is(dataId)) {
      final Collection<CommonModelElement> elements = getSelectedElements(diagram);
      if (!elements.isEmpty()) {
        return new JamDeleteProvider(new DefaultUserResponse(diagram.getProject()), elements);
      }
    }
    else if (CommonDataKeys.NAVIGATABLE_ARRAY.is(dataId)) {
      final Collection<CommonModelElement> elements = getSelectedElements(diagram);
      if (!elements.isEmpty()) {
        final HashSet<Navigatable> result = new HashSet<>();
        for (CommonModelElement element : elements) {
          final PsiElement psiElement = element.getIdentifyingPsiElement();
          if (psiElement instanceof Navigatable) {
            result.add((Navigatable)psiElement);
          }
          else if (psiElement != null) {
            final PsiFile containingFile = psiElement.getContainingFile();
            final VirtualFile file = containingFile == null ? null : containingFile.getVirtualFile();
            if (file != null) {
              result.add(PsiNavigationSupport.getInstance().createNavigatable(psiElement.getProject(), file,
                                                                              psiElement.getTextOffset()));
            }
          }
        }
        return result.toArray(Navigatable.EMPTY_NAVIGATABLE_ARRAY);
      }
    }
    return null;
  }

  private Collection<CommonModelElement> getSelectedElements(final PersistenceDiagram<PersistencePackage, PersistentObject, PersistentAttribute> diagram) {
    final Collection<CommonModelElement> elements = new HashSet<>();
    diagram.processSelectedNodes((persistentObject, persistentAttribute) -> {
      ContainerUtil.addIfNotNull(elements, persistentObject);
      return true;
    });
    diagram.processSelectedEdges((persistentObject, persistentAttribute) -> {
      ContainerUtil.addIfNotNull(elements, persistentAttribute);
      if (persistentAttribute instanceof PersistentRelationshipAttribute) {
        elements.addAll(myModelBrowser.queryTheOtherSideAttributes((PersistentRelationshipAttribute)persistentAttribute, false).findAll());
      }
      return true;
    });
    return elements;
  }

  @Override
  public boolean processEditNode(final PersistenceDiagram<PersistencePackage, PersistentObject, PersistentAttribute> diagram, final PersistentObject entity) {
    return false;
  }

  @Override
  public boolean processEditEdge(final PersistenceDiagram<PersistencePackage, PersistentObject, PersistentAttribute> persistenceDiagram) {
    return false;
  }

  @Override
  public void processCreateEdge(final PersistenceDiagram<PersistencePackage, PersistentObject, PersistentAttribute> persistenceDiagram,
                                final PersistentObject sourceEntity, final PersistentObject targetEntity) {
  }

  @Override
  public void customizeGraphView(final Graph2DView view, final EditMode editMode) {
    // nothing
  }

  public static void initUpdateListenerOnFacet(@NotNull final GraphBuilder builder, @NotNull final Facet facet) {
    FacetModificationTrackingService.getInstance(facet).addModificationTrackerListener(facet, new ModificationTrackerListener<>() {
      @Override
      public void modificationCountChanged(@NotNull final Facet facet) {
        if (builder.getView().getJComponent().isShowing()) {
          builder.queueUpdate();
        }
      }
    }, builder);
    EditorFactory.getInstance().getEventMulticaster().addDocumentListener(new DocumentListener() {
      @Override
      public void documentChanged(@NotNull final DocumentEvent event) {
        if (builder.getView().getJComponent().isShowing()) {
          builder.queueUpdate();
        }
      }
    }, builder);
  }
}