package com.intellij.psi.css.descriptor;

import com.intellij.openapi.util.NlsContexts;
import com.intellij.psi.PsiElement;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;

public interface CssElementDescriptor {
  /**
   * @return CSS version where element had been specified initially
   */
  @NotNull
  CssVersion getCssVersion();

  /**
   * @return browser versions that supported element
   */
  BrowserVersion @NotNull [] getBrowsers();

  /**
   * @return true if the element can be used in context of given ar-rule.
   */
  boolean isAllowedInContextType(@NotNull CssContextType contextType);

  /**
   * @return identifier of descriptor, not really uniq though. Use identifier for completion, descriptors retrieving, etc.
   */
  @NotNull
  String getId();

  /**
   * @return presentable name of descriptor. Use it for all UI representation of descriptor (quick doc, param info, tree view, etc.)
   */
  @NotNull
  String getPresentableName();

  @NotNull
  String getDescription();

  @Nullable
  String getDocumentationString(@Nullable PsiElement context);

  @NotNull @NlsContexts.Label
  String getElementTypeName();

  @Nullable
  String getSpecificationUrl();

  CssContextType @NotNull [] getAllowedContextTypes();

  @Nullable
  Icon getIcon();

  enum CssVersion {
    UNKNOWN("UNKNOWN", 0),
    CSS_1("CSS 1.0", 1),
    CSS_2("CSS 2.0", 2),
    CSS_2_1("CSS 2.1", 3),
    CSS_3_0("CSS 3.0", 4),
    CSS_4_0("CSS 4.0", 5),
    @Deprecated CSS_3_0_MOZILLA("CSS3 Mozilla", 0), //use BrowserVersion
    @Deprecated CSS_3_0_SAFARI("CSS3 Safari", 0), //use BrowserVersion
    @Deprecated CSS_3_0_OPERA("CSS3 Opera", 0);// use BrowserVersion

    @NotNull private final String myPresentableName;
    private final int myValue;

    CssVersion(@NotNull String presentableName, int myValue) {
      myPresentableName = presentableName;
      this.myValue = myValue;
    }

    @NotNull
    public String getPresentableName() {
      return myPresentableName;
    }

    @NotNull
    public static CssVersion fromString(@Nullable String string) {
      if ("1.0".equals(string)) {
        return CSS_1;
      }
      else if ("2.0".equals(string)) {
        return CSS_2;
      }
      else if ("2.1".equals(string)) {
        return CSS_2_1;
      }
      else if ("3.0".equals(string)) {
        return CSS_3_0;
      }
      else if ("4.0".equals(string)) {
        return CSS_4_0;
      }
      return UNKNOWN;
    }

    public int value() {
      return myValue;
    }
  }
}
