package com.intellij.lang.css;

import com.intellij.openapi.extensions.ExtensionPointName;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.NlsContexts;
import com.intellij.psi.css.CssBundle;
import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;

public abstract class CssDialect {
  public static final ExtensionPointName<CssDialect> EP_NAME = ExtensionPointName.create("com.intellij.css.dialect");
  public static final CssDialect CLASSIC = new CssDialect() {
    @Override
    public @NotNull String getName() {
      return "CLASSIC";
    }

    @Override
    public @NotNull @NlsContexts.Label String getDisplayName() {
      return CssBundle.message("css.w3c.dialect.name");
    }

    @Override
    public boolean isDefault(@NotNull Module module) {
      return true;
    }
  };

  public abstract @NotNull String getName();

  public abstract @NotNull @NlsContexts.Label String getDisplayName();

  public abstract boolean isDefault(@NotNull Module module);

  @Contract("null -> null")
  public static @Nullable CssDialect getDefaultDialect(@Nullable Module module) {
    if (module != null) {
      for (CssDialect dialect : EP_NAME.getExtensionList()) {
        if (dialect.isDefault(module)) return dialect;
      }
    }
    return null;
  }

  public static @NotNull List<CssDialect> getAvailableDialects() {
    final List<CssDialect> results = new ArrayList<>(EP_NAME.getExtensionList());
    results.add(CLASSIC);
    return results;
  }

  public static @NotNull List<CssDialect> getCustomDialects() {
    return new ArrayList<>(EP_NAME.getExtensionList());
  }

  public static @NotNull CssDialect getDialectByName(@NotNull String dialectName) {
    if (CLASSIC.getName().equals(dialectName)) {
      return CLASSIC;
    }
    CssDialect dialect = EP_NAME.getByKey(dialectName, CssDialect.class, d -> d.getName());
    return dialect != null ? dialect : CLASSIC;
  }
}
