// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.toolWindow;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleManager;
import com.intellij.openapi.module.ModuleType;
import com.intellij.openapi.project.DumbService;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.roots.ui.configuration.ModulesAlphaComparator;
import com.intellij.openapi.roots.ui.configuration.ModulesConfigurator;
import com.intellij.spring.facet.SpringFacet;
import com.intellij.spring.facet.SpringFileSetService;
import com.intellij.spring.model.utils.SpringCommonUtils;
import com.intellij.spring.settings.SpringGeneralSettings;
import com.intellij.ui.FinderRecursivePanel;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.swing.*;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

public abstract class SpringModulesPanelBase extends FinderRecursivePanel<Module> {

  protected SpringModulesPanelBase(Project project, String groupId) {
    super(project, groupId);
  }

  @NotNull
  @Override
  public List<Module> getListItems() {
    boolean autoConfigurationMode = SpringGeneralSettings.getInstance(getProject()).isAllowAutoConfigurationMode();

    if (!SpringCommonUtils.hasSpringFacets(getProject()) && !autoConfigurationMode) {
      return Collections.emptyList();
    }

    List<Module> items = new ArrayList<>();
    Module[] modules = ModuleManager.getInstance(getProject()).getModules();
    for (Module module : modules) {
      if (autoConfigurationMode || SpringCommonUtils.hasSpringFacet(module)) {
        items.add(module);
      }
    }
    items.sort(ModulesAlphaComparator.INSTANCE);
    return items;
  }

  @NotNull
  @Override
  public String getItemText(@NotNull Module module) {
    return module.getName();
  }

  @Nullable
  @Override
  public Icon getItemIcon(@NotNull Module module) {
    return module.isDisposed() ? null : ModuleType.get(module).getIcon();
  }

  @Override
  public boolean performEditAction() {
    final Module module = getSelectedValue();
    if (module != null) {
      ModulesConfigurator.showDialog(getProject(), module.getName(), null);
    }
    return true;
  }

  @Override
  public boolean hasChildren(@NotNull Module module) {
    if (module.isDisposed()) return false;
    if (DumbService.isDumb(getProject())) return false;

    final SpringFacet springFacet = SpringFacet.getInstance(module);
    if (springFacet == null) return false;
    return !SpringFileSetService.getInstance().getAllSets(springFacet).isEmpty();
  }
}