/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.spring.toolWindow;

import com.intellij.ProjectTopics;
import com.intellij.facet.ProjectWideFacetAdapter;
import com.intellij.facet.ProjectWideFacetListener;
import com.intellij.facet.ProjectWideFacetListenersRegistry;
import com.intellij.ide.DataManager;
import com.intellij.ide.actions.ContextHelpAction;
import com.intellij.openapi.Disposable;
import com.intellij.openapi.actionSystem.AnAction;
import com.intellij.openapi.actionSystem.DataContext;
import com.intellij.openapi.actionSystem.PlatformDataKeys;
import com.intellij.openapi.application.AppUIExecutor;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.project.ModuleListener;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.roots.ModuleRootEvent;
import com.intellij.openapi.roots.ModuleRootListener;
import com.intellij.openapi.ui.SimpleToolWindowPanel;
import com.intellij.openapi.util.Disposer;
import com.intellij.openapi.wm.ToolWindow;
import com.intellij.openapi.wm.ToolWindowManager;
import com.intellij.spring.facet.SpringFacet;
import com.intellij.ui.FinderRecursivePanel;
import com.intellij.ui.content.Content;
import com.intellij.ui.content.ContentManager;
import com.intellij.util.Function;
import com.intellij.util.messages.MessageBusConnection;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

/**
 * @author Yann C&eacute;bron
 */
public abstract class SpringBaseView extends SimpleToolWindowPanel implements Disposable {

  protected final Project myProject;

  protected FinderRecursivePanel<?> myRootPanel;

  protected SpringBaseView(Project project) {
    super(false, true);
    myProject = project;

    refreshContentPanel();
  }

  protected abstract FinderRecursivePanel<?> createRootPanel();

  private void refreshContentPanel() {
    myRootPanel = createRootPanel();
    myRootPanel.initPanel();
    setContent(myRootPanel);

    Disposer.register(this, myRootPanel);
  }

  /**
   * Invoked from Project/Facet settings.
   */
  protected void performFullUpdate() {
    AppUIExecutor.onUiThread().expireWith(this).submit(() -> {
      FinderRecursivePanel<?> oldPanel = myRootPanel;
      remove(oldPanel);
      Disposer.dispose(oldPanel);

      refreshContentPanel();
    });
  }

  protected void performDetailsUpdate() {
    FinderRecursivePanel<?> panel = myRootPanel;
    while (true) {
      if (!(panel.getSecondComponent() instanceof FinderRecursivePanel)) {
        panel.updateRightComponent(true);
        break;
      }
      panel = (FinderRecursivePanel<?>)panel.getSecondComponent();
    }
  }

  private void updateSelectedPath(Object... pathToSelect) {
    myRootPanel.updateSelectedPath(pathToSelect);
  }

  protected MessageBusConnection installProjectModuleListener() {
    ProjectWideFacetListener<? extends SpringFacet> myProjectWideFacetAdapter = new ProjectWideFacetAdapter<>() {
      @Override
      public void facetAdded(@NotNull SpringFacet facet) {
        performFullUpdate();
      }

      @Override
      public void facetRemoved(@NotNull SpringFacet facet) {
        performFullUpdate();
      }

      @Override
      public void facetConfigurationChanged(@NotNull SpringFacet facet) {
        performFullUpdate();
      }
    };
    ProjectWideFacetListenersRegistry.getInstance(myProject)
      .registerListener(SpringFacet.FACET_TYPE_ID, myProjectWideFacetAdapter, this);


    MessageBusConnection messageBusConnection = myProject.getMessageBus().connect(this);
    messageBusConnection.subscribe(ProjectTopics.PROJECT_ROOTS, new ModuleRootListener() {
      @Override
      public void rootsChanged(@NotNull ModuleRootEvent event) {
        performFullUpdate();
      }
    });
    messageBusConnection.subscribe(ProjectTopics.MODULES, new ModuleListener() {
      @Override
      public void moduleAdded(@NotNull Project project, @NotNull Module module) {
        performFullUpdate();
      }

      @Override
      public void moduleRemoved(@NotNull Project project, @NotNull Module module) {
        performFullUpdate();
      }

      @Override
      public void modulesRenamed(@NotNull Project project,
                                 @NotNull List<Module> modules,
                                 @NotNull Function<Module, String> oldNameProvider) {
        performFullUpdate();
      }
    });
    return messageBusConnection;
  }

  /**
   * Performs recursive update for given selection path.
   *
   * @param project      Project.
   * @param pathToSelect Path to select.
   * @param requestFocus Focus requested.
   * @param tabName      Tab to select ({@link SpringToolWindowContent#getDisplayName}).
   */
  protected static void select(Project project, final Object[] pathToSelect, final boolean requestFocus, final String tabName) {
    final ToolWindow toolWindow = ToolWindowManager.getInstance(project).getToolWindow(SpringToolWindowContent.TOOLWINDOW_ID);
    assert toolWindow != null;

    Runnable runnable = () -> {
      ContentManager contentManager = toolWindow.getContentManager();
      Content content = contentManager.findContent(tabName);

      if (content != null) {
        contentManager.setSelectedContentCB(content, requestFocus).doWhenDone(() -> {
          SpringBaseView springBaseView = (SpringBaseView)content.getComponent();
          springBaseView.updateSelectedPath(pathToSelect);
        });
      }
      else {
        // Tool window's content hasn't been initialized asynchronously yet.
        DataContext context = DataManager.getInstance().getDataContext(contentManager.getComponent());
        SpringToolWindowContentUpdater updater = SpringToolWindowContent.CONTENT_UPDATER.getData(context);
        assert updater != null;

        updater.update(() -> {
          Content updatedContent = contentManager.findContent(tabName);
          if (updatedContent == null) {
            // Corresponding tab is not available.
            return;
          }
          contentManager.setSelectedContentCB(updatedContent, requestFocus).doWhenDone(() -> {
            // Invoke active() again to avoid focus moving from selected path to the first column,
            // because tool window's FocusTask requests focus alarm with delay if component is not created.
            toolWindow.activate(() -> {
              SpringBaseView springBaseView = (SpringBaseView)updatedContent.getComponent();
              springBaseView.updateSelectedPath(pathToSelect);
            });
          });
        });
      }
    };
    if (requestFocus) {
      toolWindow.activate(runnable);
    }
    else {
      runnable.run();
    }
  }

  @Nullable
  @Override
  public Object getData(@NotNull String dataId) {
    if (PlatformDataKeys.HELP_ID.is(dataId)) {
      return getHelpId();
    }
    return super.getData(dataId);
  }

  @Override
  public void dispose() {
  }

  /**
   * @return Help ID for this view.
   */
  protected String getHelpId() {
    return "Reference.Spring.ToolWindow";
  }

  /**
   * @return Help action for use in toolbar.
   */
  protected AnAction getHelpAction() {
    return new ContextHelpAction(getHelpId());
  }
}
