// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.spi;

import com.intellij.lang.properties.IProperty;
import com.intellij.lang.properties.psi.impl.PropertiesFileImpl;
import com.intellij.openapi.module.Module;
import com.intellij.psi.PsiClass;
import com.intellij.util.PairProcessor;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

/**
 * Provides access to {@code spring.factories} configuration files.
 */
public abstract class SpringSpiManager {

  public static SpringSpiManager getInstance(Module module) {
    return module.getService(SpringSpiManager.class);
  }

  /**
   * Returns all configuration files in current module.
   *
   * @param includeTests Include config files in test source roots.
   * @return List of files.
   */
  public abstract List<PropertiesFileImpl> getSpringFactoriesFiles(boolean includeTests);

  /**
   * Returns all resolved classes configured for given key.
   *
   * @param includeTests Include config files in test source roots.
   * @param key          Key to search for.
   * @return Resolved classes or empty list of none found.
   */
  public abstract List<PsiClass> getClassesListValue(boolean includeTests, @NotNull String key);

  /**
   * Process all resolved classes list values.
   *
   * @param includeTests Process config files in test source roots.
   * @param processor    Processor.
   * @param valueHint    Only process properties containing given text (class FQN) as value or {@code null} to process all properties.
   * @return Processing result.
   */
  public abstract boolean processClassesListValues(boolean includeTests,
                                                   @Nullable String valueHint,
                                                   PairProcessor<IProperty, PsiClass> processor);
}
