// Copyright 2000-2019 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.references;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.TextRange;
import com.intellij.psi.JavaPsiFacade;
import com.intellij.psi.PsiClass;
import com.intellij.psi.PsiElement;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.utils.SpringModelSearchers;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * Reference to Spring Bean by name, optionally limited to required base-class.
 */
public class SpringBeanResourceReference extends SpringBeanReference {


  public SpringBeanResourceReference(final PsiElement element,
                                     TextRange range,
                                     @Nullable final PsiClass requiredClass,
                                     boolean isFactoryBeanRef) {
    super(element, range,requiredClass, isFactoryBeanRef);

  }

  @Override
  public PsiElement resolve() {
    final String beanName = getValue();

    final CommonSpringModel springModel = getSpringModel();

     SpringBeanPointer pointer = SpringModelSearchers.findBean(springModel, beanName);
    return pointer == null || !pointer.isValid() ?
           resolveResourceByFqn(beanName, springModel, getElement().getProject()) : pointer.getPsiElement();
  }

  @Nullable
  public static PsiClass resolveResourceByFqn(@NotNull String beanName, @NotNull CommonSpringModel model, @NotNull Project project) {
    Module module = model.getModule();
    GlobalSearchScope scope = module != null ? GlobalSearchScope.moduleWithDependenciesAndLibrariesScope(module) : GlobalSearchScope.allScope(project);
    return JavaPsiFacade.getInstance(project).findClass(beanName.replace("$", "."), scope);
  }

  @Override
  public boolean isSoft() {
    return true;
  }
}
