// Copyright 2000-2019 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.references;

import com.intellij.codeInsight.daemon.EmptyResolveMessageProvider;
import com.intellij.codeInsight.lookup.LookupElement;
import com.intellij.openapi.util.TextRange;
import com.intellij.psi.*;
import com.intellij.spring.CommonSpringModel;
import com.intellij.spring.SpringApiBundle;
import com.intellij.spring.model.SpringBeanPointer;
import com.intellij.spring.model.SpringModelSearchParameters;
import com.intellij.spring.model.converters.SpringConverterUtil;
import com.intellij.spring.model.utils.SpringModelSearchers;
import com.intellij.spring.model.utils.SpringModelUtils;
import com.intellij.util.ArrayUtil;
import com.intellij.util.IncorrectOperationException;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * Reference to Spring Bean by name, optionally limited to required base-class.
 */
public class SpringBeanReference extends PsiReferenceBase<PsiElement> implements EmptyResolveMessageProvider {

  @Nullable
  private final PsiClass myRequiredClass;
  private final boolean myFactoryBeanRef;

  public SpringBeanReference(final PsiElement element) {
    this(element, ElementManipulators.getValueTextRange(element));
  }

  public SpringBeanReference(final PsiElement element, final TextRange range) {
    this(element, range, null, false);
  }

  public SpringBeanReference(final PsiElement element,
                             TextRange range,
                             @Nullable final PsiClass requiredClass,
                             boolean isFactoryBeanRef) {
    super(element, range);
    myRequiredClass = requiredClass;
    myFactoryBeanRef = isFactoryBeanRef;
  }

  public boolean isFactoryBeanRef() {
    return myFactoryBeanRef;
  }

  @NotNull
  protected CommonSpringModel getSpringModel() {
    return SpringModelUtils.getInstance().getSpringModel(getElement());
  }

  @Override
  public PsiElement resolve() {
    final String beanName = getValue();

    final CommonSpringModel springModel = getSpringModel();

    final SpringBeanPointer pointer = SpringModelSearchers.findBean(springModel, beanName);
    return pointer == null || !pointer.isValid() ? null : pointer.getPsiElement();
  }

  @Override
  public PsiElement bindToElement(@NotNull final PsiElement element) throws IncorrectOperationException {
    return getElement();
  }

  @Override
  public Object @NotNull [] getVariants() {
    final CommonSpringModel model = getSpringModel();

    final Collection<SpringBeanPointer> beans;
    if (myRequiredClass != null && !CommonClassNames.JAVA_LANG_OBJECT.equals(myRequiredClass.getQualifiedName())) {
      final SpringModelSearchParameters.BeanClass searchParameters =
        SpringModelSearchParameters.byClass(myRequiredClass).withInheritors().effectiveBeanTypes();
      beans = SpringModelSearchers.findBeans(model, searchParameters);
    }
    else {
      beans = model.getAllCommonBeans();
    }

    List<LookupElement> lookups = new ArrayList<>(beans.size());
    for (SpringBeanPointer bean : beans) {
      ContainerUtil.addIfNotNull(lookups, SpringConverterUtil.createCompletionVariant(bean));
    }
    return ArrayUtil.toObjectArray(lookups);
  }

  @Override
  @NotNull
  public String getUnresolvedMessagePattern() {
    return SpringApiBundle.message("model.bean.error.message", getValue());
  }
}
