// Copyright 2000-2019 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.xml.beans;

import com.intellij.jam.model.common.CommonModelElement;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiElement;
import com.intellij.spring.model.SpringProfile;
import com.intellij.spring.model.converters.SpringProfileConverter;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.xml.Convert;
import com.intellij.util.xml.GenericAttributeValue;
import org.jetbrains.annotations.NotNull;

import java.util.*;

/**
 * @author Sergey Vasiliev
 */
@Convert(SpringProfileConverter.class)
public abstract class SpringDomProfile implements GenericAttributeValue<List<String>>, SpringProfile, CommonModelElement {

  @Override
  public PsiElement getIdentifyingPsiElement() {
    return getXmlElement();
  }

  @NotNull
  @Override
  public Set<String> getExpressions() {
    List<String> profileNames = getValue();
    if (ContainerUtil.isEmpty(profileNames)) {
      return Collections.singleton(DEFAULT_PROFILE_NAME);
    }

    return new LinkedHashSet<>(profileNames);
  }

  @NotNull
  @Override
  public Set<String> getNames() {
    Set<String> result = new LinkedHashSet<>();
    for (String expression : getExpressions()) {
      expression = StringUtil.trimStart(expression, "!").trim();
      if (!expression.isEmpty()) {
        result.add(expression);
      }
    }
    return result;
  }

  @Override
  public boolean matches(@NotNull Set<String> activeProfiles) {
    Set<String> profiles = getExpressions();
    if (SpringProfile.DEFAULT_PROFILE_NAME.equals(ContainerUtil.getOnlyItem(profiles))) return true;

    for (String profile : profiles) {
      if (profile.startsWith("!")) {
        String notProfile = profile.substring(1);
        if (!activeProfiles.contains(notProfile)) {
          return true;
        }
      }
      else if (activeProfiles.contains(profile)) {
        return true;
      }
    }
    return false;
  }
}
