// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.utils;

import com.intellij.codeInsight.completion.CompletionUtilCore;
import com.intellij.ide.projectView.impl.ProjectRootsUtil;
import com.intellij.openapi.module.Module;
import com.intellij.openapi.module.ModuleUtilCore;
import com.intellij.openapi.util.text.DelimitedListProcessor;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiReference;
import com.intellij.psi.impl.source.resolve.reference.impl.providers.PackageReferenceSet;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.spring.model.jam.converters.SpringAntPatternPackageReferenceSet;
import com.intellij.spring.model.values.PlaceholderUtils;
import com.intellij.util.PairProcessor;
import com.intellij.util.SmartList;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

public class SpringReferenceUtils {
  public static PsiReference @NotNull [] getPsiPackagesReferences(@NotNull final PsiElement element,
                                                                  @Nullable final String text,
                                                                  final int startInElement) {
    return getPsiPackagesReferences(element, text, startInElement, ",; \n\t");
  }

  public static PsiReference[] getPsiPackagesReferences(@NotNull final PsiElement element,
                                                        @Nullable final String text,
                                                        final int startInElement,
                                                        String delimiters) {
    return getPsiPackagesReferences(element, text, startInElement, delimiters, getResolveScope(element));
  }

  public static PsiReference[] getPsiPackagesReferences(@NotNull final PsiElement element,
                                                        @Nullable final String text,
                                                        final int startInElement,
                                                        String delimiters, @NotNull final GlobalSearchScope scope) {
    if (text == null) {
      return PsiReference.EMPTY_ARRAY;
    }

    final List<PsiReference> list = new SmartList<>();
    new DelimitedListProcessor(delimiters) {
      @Override
      protected void processToken(final int start, final int end, final boolean delimitersOnly) {
        final String packageName = text.substring(start, end);
        if (PlaceholderUtils.getInstance().isDefaultPlaceholder(packageName)) return;
        final String trimmedPackageName = packageName.trim();
        final PackageReferenceSet referenceSet;
        int offset = start + packageName.indexOf(trimmedPackageName) + startInElement;
        if (trimmedPackageName.equals(CompletionUtilCore.DUMMY_IDENTIFIER_TRIMMED)) {
          referenceSet = new SpringAntPatternPackageReferenceSet(CompletionUtilCore.DUMMY_IDENTIFIER, element, offset, scope);
        }
        else {
          referenceSet = new SpringAntPatternPackageReferenceSet(trimmedPackageName, element, offset, scope);
        }
        list.addAll(referenceSet.getReferences());
      }
    }.processText(text);
    return list.isEmpty() ? PsiReference.EMPTY_ARRAY : list.toArray(PsiReference.EMPTY_ARRAY);
  }

  public static GlobalSearchScope getResolveScope(@NotNull PsiElement element) {
    Module module = ModuleUtilCore.findModuleForPsiElement(element);
    if (module == null) {
      return GlobalSearchScope.allScope(element.getProject());
    }

    final boolean inTestSource = ProjectRootsUtil.isInTestSource(element.getContainingFile());
    return module.getModuleRuntimeScope(inTestSource);
  }

  public static boolean processSeparatedString(final String str, final String delimiter, final PairProcessor<? super String, ? super Integer> processor) {
    if (str == null) return true;
    if (StringUtil.isEmptyOrSpaces(str)) return processor.process(str, 0);

    int pos = 0;
    while (true) {
      int index = str.indexOf(delimiter, pos);
      if (index == -1) break;
      final int nextPos = index + delimiter.length();
      String token = str.substring(pos, index);
      if (token.length() != 0) {
        if (!processor.process(token.trim(), pos + token.indexOf(token.trim()))) return false;
      }
      pos = nextPos;
    }
    if (pos < str.length()) {
      final String s = str.substring(pos);
      if (!processor.process(s.trim(), pos + s.indexOf(s.trim()))) return false;
    }
    return true;
  }
}
