// Copyright 2000-2019 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.utils;

import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.PsiManager;
import com.intellij.spring.SpringModificationTrackersManager;
import com.intellij.spring.facet.SpringFileSetService;
import com.intellij.spring.model.CommonSpringBean;
import com.intellij.spring.model.SpringProfile;
import com.intellij.spring.model.xml.beans.Beans;
import com.intellij.ui.EditorNotifications;
import com.intellij.util.SmartList;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.xml.DomUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

import static com.intellij.util.containers.ContainerUtil.isEmpty;

public class SpringProfileUtils {
  public static boolean isActiveProfile(@NotNull Beans beans, @Nullable Set<String> activeProfiles) {
    if (isEmpty(activeProfiles)) return true;

    if (beans.getProfile().matches(activeProfiles)) {
      //IDEA-67463
      Beans parentBeans = DomUtil.getParentOfType(beans, Beans.class, true);
      return parentBeans == null || isActiveProfile(parentBeans, activeProfiles);
    }
    return false;
  }

  @NotNull
  public static String profilesAsString(@Nullable Set<String> activeProfiles) {
    if (isEmpty(activeProfiles)) return "";

    Set<String> profiles = new TreeSet<>();

    for (String activeProfile : activeProfiles) {
      if (!SpringProfile.DEFAULT_PROFILE_NAME.equals(activeProfile)) {
        profiles.add(activeProfile);
      }
    }

    if (profiles.size() > 1) {
      return StringUtil.join(profiles, ", ");
    }
    return ContainerUtil.getFirstItem(profiles, "");
  }

  @NotNull
  public static Set<String> profilesFromString(@Nullable String text) {
    if (StringUtil.isEmptyOrSpaces(text)) return Collections.emptySet();

    Set<String> activeProfiles = new LinkedHashSet<>();
    String[] names = text.split(",");
    for (String name : names) {
      name = name.trim();
      if (!name.isEmpty()) {
        activeProfiles.add(name);
      }
    }
    return activeProfiles;
  }

  @NotNull
  public static <T extends CommonSpringBean> List<T> filterBeansInActiveProfiles(@NotNull Collection<? extends T> allBeans,
                                                                                 @Nullable Set<String> activeProfiles) {
    if (isEmpty(activeProfiles)) return new SmartList<>(allBeans);

    return ContainerUtil.filter(allBeans, bean -> bean.isValid() && bean.getProfile().matches(activeProfiles));
  }

  public static boolean isInActiveProfiles(@NotNull CommonSpringBean bean, @Nullable Set<String> activeProfiles) {
    return isEmpty(activeProfiles) || bean.getProfile().matches(activeProfiles);
  }

  public static void notifyProfilesChanged(@NotNull Project project) {
    SpringModificationTrackersManager.getInstance(project).fireActiveProfilesChanged();
    PsiManager.getInstance(project).dropPsiCaches();
    EditorNotifications.getInstance(project).updateAllNotifications();
    project.getMessageBus().syncPublisher(SpringFileSetService.TOPIC).activeProfilesChanged();
  }
}
