// Copyright 2000-2019 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.spring.model.jam.utils.filters;

import com.intellij.openapi.module.Module;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.psi.*;
import com.intellij.psi.search.GlobalSearchScope;
import com.intellij.psi.search.PackageScope;
import com.intellij.psi.search.searches.ClassInheritorsSearch;
import com.intellij.spring.model.jam.stereotype.CustomSpringComponent;
import com.intellij.spring.model.jam.stereotype.SpringStereotypeElement;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.Set;

public class SpringContextIncludeAssignableFilter extends SpringContextFilter.IncludeExpression {

  public SpringContextIncludeAssignableFilter(@Nullable String expression) {
    super(expression);
  }

  @NotNull
  @Override
  public Set<SpringStereotypeElement> includeStereotypes(@NotNull Module module, @NotNull Set<PsiPackage> packages) {
    final Set<SpringStereotypeElement> components = new LinkedHashSet<>();

    final String fqn = getExpression();

    if (!StringUtil.isEmptyOrSpaces(fqn)) {
      final GlobalSearchScope searchScope = GlobalSearchScope.moduleWithDependenciesAndLibrariesScope(module);
      final PsiClass assignableClass = JavaPsiFacade.getInstance(module.getProject()).findClass(fqn, searchScope);

      addCustomComponents(packages, searchScope, components, assignableClass);
    }
    return components;
  }

  public static void addCustomComponents(@NotNull Set<? extends PsiPackage> packages,
                                         @NotNull GlobalSearchScope searchScope,
                                         @NotNull Set<? super SpringStereotypeElement> components,
                                         @Nullable PsiClass assignableClass) {
    if (assignableClass == null) return;

    addCustomComponent(components, assignableClass);

    for (PsiPackage psiPackage : packages) {
      GlobalSearchScope pkgSearchScope = searchScope.intersectWith(PackageScope.packageScope(psiPackage, true));

      final Collection<PsiClass> inheritors = ClassInheritorsSearch.search(assignableClass, pkgSearchScope, true).findAll();
      for (PsiClass psiClass : inheritors) {
        addCustomComponent(components, psiClass);
      }
    }
  }

  private static void addCustomComponent(@NotNull Set<? super SpringStereotypeElement> components, @NotNull PsiClass componentCandidateClass) {
    final PsiModifierList modifierList = componentCandidateClass.getModifierList();

    if (componentCandidateClass.isInterface() ||
        componentCandidateClass.isAnnotationType() ||
        (modifierList != null && modifierList.hasModifierProperty(PsiModifier.ABSTRACT))) {
      return;
    }

    components.add(new CustomSpringComponent(componentCandidateClass));
  }
}
